# -*- coding: utf-8 -*-
from django.conf import settings

from travel.avia.library.python.common.models.currency import Price as CommonPrice


class Price(CommonPrice):
    def __init__(self, value, currency=None):
        currency = {'RUB': 'RUR'}.get(currency, currency)

        super(Price, self).__init__(value, currency)

    def __add__(self, other):
        if not isinstance(other, CommonPrice):
            raise ValueError(
                u'Can only sum prices with prices. Got %s' % type(other))

        if self.currency != other.currency:
            raise ValueError(u'Can only add prices with same currency')

        return Price(self.value + other.value, self.currency)

    def __eq__(self, other):
        if not isinstance(other, CommonPrice):
            return False
        if self.currency != other.currency:
            return False
        return self.value == other.value

    def __hash__(self):
        return hash((self.value, self.currency))

    @classmethod
    def convert_to_national(cls, tariff, national_version, rates):
        # typing: (Price, str, Any) -> Price
        try:
            to_currency = settings.AVIA_NATIONAL_CURRENCIES[national_version]
        except KeyError:
            raise ValueError('Not allowed national_version [%s]', national_version)

        return cls.convert_to_currency(tariff, to_currency, rates)

    @classmethod
    def convert_to_currency(cls, tariff, currency, rates):
        # typing: (Price, str, Any) -> Price
        if tariff.currency == currency:
            return tariff

        if not rates:
            raise ValueError('No rates')

        if tariff.currency not in rates:
            raise ValueError('Tariff currency [%s] not in rates', tariff.currency)

        if currency not in rates:
            raise ValueError('Target currency [%s] not in rates', currency)

        return cls(
            float(tariff.value) *
            float(rates[tariff.currency]) /
            float(rates[currency]),
            currency
        )
