# -*- coding: utf-8 -*-
from functools import partial, update_wrapper, wraps
from inspect import getargspec


def elementwise(fn):
    @wraps(fn)
    def wrapper(arg, *args, **kwargs):
        # is a Sequence
        if hasattr(arg, '__getitem__') and not isinstance(arg, basestring):
            return [fn(item, *args, **kwargs) for item in arg]

        else:
            return fn(arg, *args, **kwargs)

    return wrapper


def pipe(sink):
    """Пропустит результат через функцию sink"""
    def deco(fn):
        @wraps(fn)
        def wrapper(*args, **kwargs):
            return sink(fn(*args, **kwargs))
        return wrapper
    return deco


def filter_results(keyfunc=None):
    ''' Декоратор для фильтрации возвращаемого функцией iterable '''
    return pipe(partial(filter, keyfunc))


def on_exception(callback):
    def deco(fn):
        @wraps(fn)
        def wrapper(*args, **kwargs):
            try:
                return fn(*args, **kwargs)

            except Exception as e:
                try:
                    raise
                finally:
                    callback(e, fn, *args, **kwargs)

        return wrapper
    return deco


# Copy of django.utils.decorators.method_decorator
def method_decorator(dec_orig):
    """
    Converts a function decorator into a method decorator
    """
    # 'func' is a function at the time it is passed to _dec, but will
    # eventually be a method of the class it is defined it.
    def _dec(func):
        def _wrapper(self, *args, **kwargs):
            @dec_orig
            def bound_func(*args2, **kwargs2):
                return func(self, *args2, **kwargs2)
            # bound_func has the signature that 'dec_orig' expects i.e.  no
            # 'self' argument, but it is a closure over self so it can call
            # 'func' correctly.
            return bound_func(*args, **kwargs)
        # In case 'dec_orig' adds attributes to the function it decorates, we
        # want to copy those. We don't have access to bound_func in this scope,
        # but we can cheat by using it on a dummy function.

        @dec_orig
        def dummy(*args, **kwargs):
            pass
        update_wrapper(_wrapper, dummy)
        # Need to preserve any existing attributes of 'func',
        # including the name.
        update_wrapper(_wrapper, func)

        return _wrapper
    update_wrapper(_dec, dec_orig)
    # Change the name to aid debugging.
    _dec.__name__ = 'method_decorator(%s)' % dec_orig.__name__
    return _dec


def maybe_method_decorator(dec):
    m_dec = method_decorator(dec)

    @wraps(dec)
    def dec_wrapper(fn):
        spec = getargspec(fn)

        if spec.args and spec.args[0] == 'self':
            return m_dec(fn)

        else:
            return dec(fn)

    return dec_wrapper


@method_decorator
def memoize_method(fn):
    @wraps(fn)
    def memoized_fn(*args, **kwargs):
        if not hasattr(fn, '_memo'):
            fn._memo = fn(*args, **kwargs)
        return fn._memo
    return memoized_fn
