# -*- coding: utf-8 -*-
import fcntl
import os

from contextlib import contextmanager


LOCKS_DIR = os.environ.get('AVIA_TICKET_DAEMON_LOCK_DIRS', '/tmp')


def acquire_lock(key):
    """
    acquire lock for key
    create lock-file with name {key}.lock
    """
    filename = "%s/%s.lock" % (LOCKS_DIR, key.replace('/', '_'))
    open_mode = os.O_RDWR | os.O_CREAT | os.O_TRUNC
    fd = os.open(filename, open_mode)

    fcntl.flock(fd, fcntl.LOCK_EX)

    return fd


def release_lock(fd):
    """
    No deletion of file with fd to prevent race condition on creating/deleting lock-file
    """
    fcntl.flock(fd, fcntl.LOCK_UN)
    os.close(fd)


@contextmanager
def lock_resource(key):
    """
    blocking lock for acquiring resource with key
    """
    lock_fd = acquire_lock(key)

    try:
        yield
    finally:
        release_lock(lock_fd)
