# -*- coding: utf-8 -*-
import logging
from functools import partial
from itertools import chain

import gevent

from travel.avia.ticket_daemon.ticket_daemon.api.flights import FlightFabric
from travel.avia.ticket_daemon.ticket_daemon.api.result import Statuses
from travel.avia.ticket_daemon.ticket_daemon.daemon.importer import Importer
from travel.avia.ticket_daemon.ticket_daemon.lib.tracker import ExchangeMessage

logger = logging.getLogger(__name__)


class PartnerDebugger(object):
    def __call__(self, query, partner):
        gevent.spawn(self._call, query, partner)

    def _call(self, query, partner):
        flight_fabric = FlightFabric()
        importer_code = Importer.get_importer_code_for_partner(partner)
        logger.info('Importer code: %s', importer_code)
        importer_cls = Importer.get_importer_cls_by_code(importer_code)
        logger.info('Importer cls: %s', importer_cls.__name__)

        importer = importer_cls(
            code=importer_code,
            partners=[partner],
            query=query,
            response_collector=None,
            flight_fabric=flight_fabric,
            big_beauty_collectors=[],
        )
        query = importer.q
        importer.dialog.write_dialog_status(Statuses.QUERYING)
        chunked_query_function = importer.prepare_chunked_queryfun(importer_code, query)

        variants = list(chain.from_iterable(chunked_query_function(query)))
        importer.assign_partners_to_variants(variants)
        variants = self.prepare_variants(variants, partner, query)
        self.add_variants_event(variants, query)

        importer.dialog.write_dialog_exchanges()

        logger.info('Flights: %d', len(flight_fabric.get_flights()))

    def prepare_variants(self, variants, partner, query):
        variants = [v for v in variants if getattr(v, 'partner', None) and v.partner.code == partner.code]

        for variant in variants:
            for segment in variant.all_segments:
                segment.complete(partner_code=partner.code, is_charter=variant.is_charter)

        for v in variants:
            v.what_bad = []
            find_failed_check(v, query)

        return variants

    def add_variants_event(self, variants, query):
        variants_repr = '\n'.join(map(HumanVariantFormatter.format_variant, variants))
        tracker = query.trackers.values()[0]
        tracker.exchanges[-1].events.append(ExchangeMessage('variants', variants_repr, compress=False))


def find_failed_check(v, q):
    date_forward_ok = partial(v.date_forward_ok, q)
    backward_fill_if_need = partial(v.backward_fill_if_need, q)
    date_backward_ok = partial(v.date_backward_ok, q)
    segment_service_ok = partial(v.is_segment_service_ok, q)
    variant_service_ok = partial(v.variant_service_ok, q)

    order_checks = [
        v.forward_exists,
        v.forward_transfer_ok,
        v.backward_transfer_ok,
        v.departure_after_arrival_at_different_steps,
        v.price_ok,
        date_forward_ok,
        backward_fill_if_need,
        date_backward_ok,
        segment_service_ok,
        variant_service_ok,
        v.is_not_circle_variant,
    ]

    if not v.completed_ok:
        v.what_bad.append('completeness_filter')
        return

    for check in order_checks:
        if not check():
            try:
                error = check.__name__
            except AttributeError:
                error = check.func.__name__
            v.what_bad.append(error)


class HumanVariantFormatter(object):
    @staticmethod
    def tabli(*values):
        return u'\t'.join([HumanVariantFormatter.somehow_to_str(v) for v in values])

    @staticmethod
    def somehow_to_str(v):
        try:
            return str(v)
        except Exception:
            return repr(unicode(v))

    @staticmethod
    def format_segment(s):
        columns = [
            s.number,
            s.local_departure,
            s.local_arrival,
            getattr(s, 'station_from_iata', ''),
            getattr(s, 'station_to_iata', ''),
            getattr(s, 'company_iata', ''),
        ]
        columns += filter(
            None,
            [
                (
                    u'AVIA_COMPANY:(%s)' % s.avia_company
                    if getattr(s, 'avia_company', None)
                    else s.company and (u'COMPANY:([%s] %s)' % (s.company.id, s.company.title))
                ),
                getattr(s, 'fare_code', None) and u'FARE_CODE:(%s)' % s.fare_code,
                getattr(s, 'company_tariff', None) and u'TARIFF:(%s)' % s.company_tariff,
            ],
        )
        return '\t' + HumanVariantFormatter.tabli(*columns)

    @staticmethod
    def format_variant(v):
        variant_repr = list()
        variant_repr.append(
            HumanVariantFormatter.tabli(
                'charter' if getattr(v, 'charter', False) else v.klass,
                v.tariff,
                getattr(v, 'partner_code', 'partner_code not set'),
                getattr(v, 'dohop_vendor_id', ''),
                '<%s>' % v.tag,
                v.baggage,
            )
        )

        if v.what_bad:
            variant_repr.append(('*' * 7) + ' CHECK(S) FAILED : ' + ', '.join(v.what_bad) + ' ' + ('*' * 7))

        if getattr(v, 'dohop_unverified', False):
            variant_repr.append(('*' * 7) + ' UNVERIFIED dohop ' + ('*' * 7))

        variant_repr.append('forward segments:')
        for s in v.forward.segments:
            variant_repr.append(HumanVariantFormatter.format_segment(s))

        if v.backward.segments:
            variant_repr.append('backward segments:')
            for s in v.backward.segments:
                variant_repr.append(HumanVariantFormatter.format_segment(s))

        if v.order_data:
            variant_repr.append('order_data:')
            variant_repr.append(str(v.order_data))

        return '\n'.join(variant_repr)
