# -*- coding: utf-8 -*-
import logging
from abc import abstractmethod, ABCMeta
from itertools import izip

import ujson
from typing import List

from travel.avia.ticket_daemon.ticket_daemon.api.cache import shared_cache
from travel.avia.ticket_daemon.ticket_daemon.api.result import Status, cache_backends
from travel.avia.ticket_daemon.ticket_daemon.lib import feature_flags


class IPartnerStatusFetcher(object):
    __metaclass__ = ABCMeta

    @abstractmethod
    def fetch_all(self, query, partner_codes):
        pass


class PartnerStatusFetcher(IPartnerStatusFetcher):
    def __init__(self, logger, cache):
        """
        :type logger: ILogger
        :type cache: IMCR
        """

        self._logger = logger
        self._cache = cache

    def fetch_all(self, query, partner_codes):
        # type: (any, List[str]) -> dict
        """
        Достаем статусы опроса партнеров по направлению
        """
        keys = [
            Status.make_result_key(query.qkey, p)
            for p in partner_codes
        ]
        data_to_keys = self._cache.get_many(keys)
        result = {}

        for key, partner_code in izip(keys, partner_codes):
            data = data_to_keys.get(key)
            unpacked_data = None
            if data:
                try:
                    unpacked_data = self._cache.unpack(data)
                except Exception:
                    self._logger.exception(u'Restoring Status error: [%r]', data)
            result[partner_code] = unpacked_data

        return result


class PartnerResultMetaFetcher(IPartnerStatusFetcher):
    def __init__(self, logger, cache):
        """
        :type logger: ILogger
        :type cache: IMCR
        """

        self._logger = logger
        self._cache = cache

    def fetch_all(self, query, partner_codes):
        # type: (any, List[str]) -> dict
        """
        Достаем мета-информацию результатов партнеров по направлению
        """
        result = dict.fromkeys(partner_codes)
        for row in self._cache.get_many(query, partner_codes, columns=('partner_code', 'meta')):
            try:
                result[row['partner_code']] = ujson.loads(row['meta'])
            except Exception:
                if feature_flags.log_parse_meta_exceptions():
                    self._logger.exception(u'Restoring Result meta error: [%r]', row)
                else:
                    self._logger.info(u'Restoring Result meta error: [%r]', row)
        return result


partner_status_fetcher = PartnerStatusFetcher(
    logger=logging.getLogger(__name__),
    cache=shared_cache,
)


partner_result_meta_fetcher = PartnerResultMetaFetcher(
    logger=logging.getLogger(__name__),
    cache=cache_backends.ydb_cache,
)
