# -*- coding: utf-8 -*-
import logging
from abc import abstractmethod, ABCMeta

import opentracing

from travel.avia.ticket_daemon.ticket_daemon.api.models_utils.partners import get_actual_partners
from travel.avia.ticket_daemon.ticket_daemon.api.partners_context import PartnersContext
from travel.avia.ticket_daemon.ticket_daemon.lib.partner_status_fetcher import (
    partner_status_fetcher, IPartnerStatusFetcher, partner_result_meta_fetcher
)
from travel.avia.ticket_daemon.ticket_daemon.lib.yt_loggers.query_cache_hit_logger import (
    query_cache_hit_logger, IQueryCacheHitLogger
)


class IQueryPartnerProvider(object):
    __metaclass__ = ABCMeta

    @abstractmethod
    def get_partners_for(self, query, only_partners, ignore_cache, test_id):
        pass


def get_related_partners(query, mobile=None, for_init_search=True):
    if mobile is None:
        mobile = query.is_mobile

    only_rasp_partners = query.is_from_rasp and for_init_search
    return get_actual_partners(
        query.national_version, query.t_code, only_rasp_partners, mobile
    )


class QueryPartnerProvider(IQueryPartnerProvider):
    """
    Провайдер, достающий модели партнеров/вендоров, по которым будет проинициализирован запрос
    """

    def __init__(
        self, logger, partner_status_fetcher, partner_result_meta_fetcher, query_cache_hit_logger
    ):
        assert isinstance(partner_status_fetcher, IPartnerStatusFetcher)
        assert isinstance(partner_result_meta_fetcher, IPartnerStatusFetcher)
        assert isinstance(query_cache_hit_logger, IQueryCacheHitLogger)

        self._logger = logger
        self._partner_status_fetcher = partner_status_fetcher
        self._partner_result_meta_fetcher = partner_result_meta_fetcher
        self._query_cache_hit_logger = query_cache_hit_logger

    def get_partners_for(self, query, only_partners, ignore_cache, test_id):
        """
        Достать список партнеров, для которых будет проинициализирован запрос
        :type query: ticket_daemon.api.query.Query
        :type only_partners: typing.List[str]
        :type ignore_cache: bool
        :param test_id:
        :rtype: travel.avia.ticket_daemon.ticket_daemon.api.partners_context.PartnersContext
        """
        with opentracing.global_tracer().start_active_span(operation_name='Getting partners for query',
                                                           finish_on_close=True):
            return self._get_partners_for(query, only_partners, ignore_cache, test_id)

    def _get_partners_for(self, query, only_partners, ignore_cache, test_id):
        # todo это место можно с оптимизировать
        # 1) где-то кэшировать весь список партеров
        # 2) с помощью IEnabledPartnerProvider отфильтровывать выключенных партнеров
        partners = get_related_partners(query)

        # если хотят конкретных партнеров, то фильтруем по ним
        if only_partners:
            partners = filter(lambda x: x.code in only_partners, partners)

        partners_context = PartnersContext(partners, ignore_cache)
        partner_code_to_status = self._partner_status_fetcher.fetch_all(query, [p.code for p in partners])
        partners_context.set_statuses(partner_code_to_status)
        try:
            partner_result_metas = self._partner_result_meta_fetcher.fetch_all(query, [p.code for p in partners])
            partners_context.set_partners_meta(partner_result_metas)
            self._query_cache_hit_logger.log(
                query=query,
                partners=partners,
                partner_code_to_status=partner_code_to_status,
                partner_result_metas=partner_result_metas,
                ignore_cache=ignore_cache,
                test_id=test_id
            )
        except Exception:
            self._logger.exception('Exception on logging query_cache_hits')

        return partners_context


query_partner_provider = QueryPartnerProvider(
    logger=logging.getLogger(__name__),
    partner_status_fetcher=partner_status_fetcher,
    partner_result_meta_fetcher=partner_result_meta_fetcher,
    query_cache_hit_logger=query_cache_hit_logger,
)
