# -*- coding: utf-8 -*-
import six
from django_redis.client import DefaultClient
from django_redis.exceptions import ConnectionInterrupted
from redis.sentinel import Sentinel

from travel.avia.ticket_daemon.ticket_daemon.lib.utils import skip_None_values


class RedisSentinelClient(DefaultClient):
    def __init__(self, server, params, backend):
        super(RedisSentinelClient, self).__init__(server, params, backend)

        self._hosts = self._options['HOSTS']
        self.service_name = self._options['SENTINEL_SERVICE_NAME']

        self.sentinel = Sentinel(self._hosts, socket_timeout=self._options['SOCKET_TIMEOUT'])
        options = {'password': self._options['PASSWORD']} if self._options['PASSWORD'] else {}
        options['retry_on_timeout'] = True
        options['socket_timeout'] = self._options['SOCKET_TIMEOUT']
        self.master = self.sentinel.master_for(self.service_name, **options)
        self.slave = self.sentinel.slave_for(self.service_name, **options)

    def set(self, key, value, timeout=None, version=None, client=None, nx=False, xx=False):
        return self.master.set(key, value, nx=nx, xx=xx, ex=timeout)

    def get(self, key, default=None, version=None, client=None):
        return self.slave.get(key)

    def get_client(self, write=True):
        return self.master

    def get_many(self, keys, version=None, client=None):
        if not keys:
            return {}

        return skip_None_values(dict(zip(keys, self.slave.mget(keys))))

    def _set_many_ex(self, data, timeout=None, version=None, client=None):
        if not data:
            return

        if client is None:
            client = self.get_client(write=True)

        try:
            pipeline = client.pipeline()
            for key, value in six.iteritems(data):
                self.set(key, value, timeout, version=version, client=pipeline)
            pipeline.execute()
        except Exception as e:
            raise ConnectionInterrupted(connection=client, parent=e)

    def set_many(self, data, timeout=None, version=None, client=None):
        if timeout:
            return self._set_many_ex(data, timeout=timeout, version=version, client=client)
        return self.master.mset(data)
