# -*- coding: utf-8 -*-
import json
import logging
import uuid
from datetime import datetime
from random import random

from django.conf import settings

from travel.avia.ticket_daemon.ticket_daemon.api.query import Query
from travel.avia.ticket_daemon.ticket_daemon.daemon.celery_revise_app import celery_revise_app
from travel.avia.ticket_daemon.ticket_daemon.lib.utils import wrap

log = logging.getLogger(__name__)


def should_register_redirect(partner, order_data, user_info, additional_params=None):
    national_version = order_data.get('national_version', '')

    review_percent = partner.get_review_percent(national_version)
    if not review_percent:
        return False

    is_staff = user_info.get('django_user', {}).get('is_staff', False)
    if is_staff or settings.REVISE_FORCE:
        return True

    review_percent = review_percent / 100.0
    return random() <= review_percent


def register_redirect(
    user_info, order_data, partner,
    redirect_url, post_data=None,
    utm_source=None,
    query_source=None,
    additional_data=None,
):
    if not settings.REGISTER_REDIRECT:
        return

    log.debug('Begin register_redirect')
    if not should_register_redirect(
        partner, order_data, user_info, additional_data
    ):
        log.debug(u'Skip registration [%s] user_info: %s', partner, user_info)
        return

    log.info(u'Registrate [%s] user_info: %s', partner, user_info)

    walk = CrossWalk(
        hit_time=datetime.now(),
        partner=partner,
        url=redirect_url,
        tariff=order_data['tariff'],
        shown_tariff=order_data.get('shown_tariff'),
        user_info=user_info,
        post_data=post_data,
        qkey=order_data.get('qkey'),
        utm_source=utm_source,
        query_source=query_source,
        additional_data=additional_data,
    )

    # Выполнять проверки переходов не старше 5 минут
    try:
        message_id = uuid.uuid4()
        celery_revise_app.send_task('do_review', args=(walk.serialize(), message_id,), expires=60 * 5)
        log.info(u'message sent to revise, id: %s', message_id)
    except Exception as e:
        log.error(u'failed sending message to revise %r', e)


class CrossWalk(object):
    """
    Переход к партнёру со страницы покупки
    """

    def __init__(
            self, hit_time, partner, url, tariff, shown_tariff, user_info,
            post_data, qkey, utm_source, query_source, additional_data
    ):
        self._hit_time = hit_time
        self._partner = partner
        self._url = url
        self._tariff = tariff
        self._shown_tariff = shown_tariff
        self._user_info = user_info
        self._post_data = post_data
        self._qkey = qkey
        self._utm_source = utm_source
        self._query_source = query_source
        self._additional_data = additional_data or {}

    def prepare_review(self, q=None):
        redirect_params = {'url': self._url, 'post': self._post_data}

        r = wrap({})

        r.partner = self._partner.code
        r.hit_time = self._hit_time
        r.review_time = datetime.now()
        r.order_content = self._qkey or ''
        r.redirect_params = json.dumps(redirect_params, indent=4)
        r.user_info = json.dumps(self._user_info, indent=4)
        r.price_value, r.price_currency, r.price_unixtime = parse_tariff(self._tariff)

        r.query_source = ''
        r.utm_source = ''
        r.utm_campaign = ''
        r.utm_medium = ''
        r.utm_content = ''
        r.wizard_redir_key = ''
        r.wizard_flags = ''

        if q:
            r.point_from = q.point_from.point_key
            r.point_to = q.point_to.point_key
            r.date_forward = q.date_forward
            r.date_backward = q.date_backward
            r.adults = q.passengers.get('adults')
            r.children = q.passengers.get('children')
            r.infants = q.passengers.get('infants')
            r.national_version = q.national_version
            r.klass = q.klass
            r.utm_source = self._utm_source
            r.query_source = self._query_source

        if self._shown_tariff:
            r.shown_price_value, r.shown_price_currency, r.shown_price_unixtime = parse_tariff(self._shown_tariff)

        if self._additional_data:
            r.utm_source = self._additional_data.get('utm_source')
            r.utm_campaign = self._additional_data.get('utm_campaign')
            r.utm_medium = self._additional_data.get('utm_medium')
            r.utm_content = self._additional_data.get('utm_content')
            r.wizard_redir_key = self._additional_data.get('wizard_redir_key')
            r.wizard_flags = self._additional_data.get('wizard_flags')

            # TODO: убрать обработку этих параметров, когда на фронте отпилят (TRAVELFRONT-2871)
            if r.wizard_redir_key and not self._shown_tariff:
                r.shown_price_value = self._additional_data.get('revise_price_value')
                r.shown_price_currency = self._additional_data.get('revise_price_currency')
                r.shown_price_unixtime = self._additional_data.get('revise_price_unixtime')

            if 'marker' in self._additional_data:
                r.marker = self._additional_data['marker']

        time_params = [
            'hit_time', 'review_time', 'date_forward', 'date_backward',
            'price_unixtime', 'shown_price_unixtime'
        ]
        for param in time_params:
            if not r[param]:
                continue
            if isinstance(r[param], int):
                r[param] = datetime.fromtimestamp(r[param])
            r[param] = r[param].isoformat()
        return r

    def serialize(self):
        """
        :return: json string
        """
        q = self._qkey and Query.from_key(self._qkey, service='revise')
        walk = self.prepare_review(q)

        return json.dumps(dict(walk))


def parse_tariff(tariff):
    price_value = float(tariff['value'])
    price_currency = 'RUR' if tariff['currency'] == 'RUB' else tariff['currency']

    price_unixtime = tariff.get('price_unixtime')
    if price_unixtime is not None:
        price_unixtime = int(price_unixtime)

    return price_value, price_currency, price_unixtime
