# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import logging
from datetime import datetime
from typing import List, Tuple, Optional, Any

from six import text_type, binary_type, ensure_str, ensure_text

from travel.avia.ticket_daemon.ticket_daemon.lib.baggage import Baggage, BaggageParser
from travel.avia.ticket_daemon.ticket_daemon.lib.currency import Price

logger = logging.getLogger(__name__)


class SirenaVariantBuilder(object):
    CURRENCIES_MAP = {
        u'РУБ': 'RUR',
        u'ЕВР': 'EUR',
    }

    __slots__ = ('price', 'brand', 'joints',)

    def __init__(self, price, currency, brand):
        # type: (float, text_type, text_type)->None
        self.price = Price(price, SirenaVariantBuilder.CURRENCIES_MAP[currency])
        self.brand = brand or ''
        self.joints = []  # type: List[SirenaJointBuilder]

    def add_flight(self, joint_id, flight):
        # type: (text_type, SirenaFlight)->None
        # Use it with care: joints are kept in lists to save on memory allocations:
        # don't add flights after adding Variant to set or hash
        self._get_or_create_joint(to_text_type(joint_id)).add_flight(flight)

    def _get_or_create_joint(self, joint_id):
        # type: (text_type)->SirenaJointBuilder
        if not self.joints or self.joints[-1].joint_id != joint_id:
            self.joints.append(SirenaJointBuilder(joint_id))
        return self.joints[-1]

    def build(self):
        # type: ()->SirenaVariant
        joints = tuple(joint.build() for joint in self.joints)  # type: Tuple[SirenaJoint]
        return SirenaVariant(
            self.price,
            self.brand,
            joints,
        )


class SirenaJointBuilder(object):
    __slots__ = ('joint_id', 'flights')

    def __init__(self, joint_id):
        # type: (text_type)->None
        self.joint_id = to_text_type(joint_id)
        self.flights = []

    def add_flight(self, flight):
        # type: (SirenaFlight)->None
        self.flights.append(flight)

    def build(self):
        # type: ()->SirenaJoint
        return SirenaJoint(
            self.joint_id,
            tuple(flight for flight in self.flights),
        )


class SirenaVariant(object):
    __slots__ = ('_price', '_brand', '_joints',)

    def __init__(self, price, brand, joints):
        # type: (Price, text_type, Tuple[SirenaJoint])->None
        self._price = price
        self._brand = to_text_type(brand or '')
        self._joints = joints

    @property
    def price(self):
        # type:()->Price
        return self._price

    @property
    def brand(self):
        # type:()->text_type
        return self._brand

    @property
    def joints(self):
        # type:()->Tuple[SirenaJoint]
        return self._joints

    def route_key(self):
        # type: (SirenaVariant)->Tuple[Any]
        key = tuple(joint.route_key() for joint in self.joints)
        return key

    def __unicode__(self):
        return slotted__unicode__(self)

    def __repr__(self):
        return slotted__repr__(self)

    def __eq__(self, other):
        # type: (SirenaVariant)->bool
        if not self.__class__ == other.__class__:
            raise NotImplementedError
        # skip brand intentionally
        return (
            self.price == other.price and
            self.joints == other.joints and
            all(j1 == j2 for j1, j2 in zip(self.joints, other.joints))
        )

    def __hash__(self):
        return hash((self.price, self.joints))

    def joint_id(self, origin, destination):
        # type: (text_type, text_type)->Optional[text_type]
        for joint in self.joints:  # type: SirenaJoint
            if joint.has_segment(origin, destination):
                return joint.joint_id
        return None


class SirenaJoint(object):
    __slots__ = ('_joint_id', '_flights')

    def __init__(self, joint_id, flights):
        # type: (text_type, Tuple[SirenaFlight])->None
        self._joint_id = joint_id
        self._flights = flights

    @property
    def joint_id(self):
        return self._joint_id

    @property
    def flights(self):
        # type: ()->Tuple[SirenaFlight]
        return self._flights

    def route_key(self):
        # type: ()->Tuple[Any]
        key = tuple(flight.route_key() for flight in self.flights)
        return key

    def __eq__(self, other):
        # type:(SirenaJoint)->bool
        if not self.__class__ == other.__class__:
            raise NotImplementedError
        return self.flights == other.flights

    def __hash__(self):
        return hash(self.flights)

    def __unicode__(self):
        return slotted__unicode__(self)

    def __repr__(self):
        return slotted__repr__(self)

    def has_segment(self, origin, destination):
        # type: (text_type,text_type)->bool
        origin, destination = to_text_type(origin), to_text_type(destination)
        for flight in self.flights:  # type: SirenaFlight
            if flight.origin == origin and flight.destination == destination:
                return True
        return False


class SirenaFlight(object):
    __slots__ = (
        '_company', '_num',
        '_origin', '_destination',
        '_departure', '_arrival',
        '_class_', '_subclass', '_baggage', '_fare_code'
    )

    def __init__(self, company, num, origin, destination, departure_date, departure_time, arrival_date, arrival_time,
                 class_, subclass, baggage, fare_code):
        self._company = to_text_type(company)
        self._num = to_text_type(num)
        self._origin = to_text_type(origin)
        self._destination = to_text_type(destination)
        self._departure = parse_dt(departure_date, departure_time)
        self._arrival = parse_dt(arrival_date, arrival_time)
        self._class_ = to_text_type(class_)
        self._subclass = to_text_type(subclass)
        self._baggage = parse_baggage(baggage)
        self._fare_code = to_text_type(fare_code)

    @property
    def company(self):
        # type:()->text_type
        return self._company

    @property
    def num(self):
        # type:()->text_type
        return self._num

    @property
    def origin(self):
        # type: ()->text_type
        return self._origin

    @property
    def destination(self):
        # type: ()->text_type
        return self._destination

    @property
    def departure(self):
        # type: ()->datetime
        return self._departure

    @property
    def arrival(self):
        # type: ()->datetime
        return self._arrival

    @property
    def class_(self):
        # type: ()->text_type
        return self._class_

    @property
    def subclass(self):
        # type: ()->text_type
        return self._subclass

    @property
    def baggage(self):
        # type: ()->Baggage
        return self._baggage

    @property
    def fare_code(self):
        # type: ()->text_type
        return self._fare_code

    def route_key(self):
        # type: ()-> Tuple[text_type, text_type, text_type, text_type, datetime, datetime, text_type]
        key = self.company, self.num, self.origin, self.destination, self.departure, self.arrival, self.class_
        return key

    def __unicode__(self):
        return slotted__unicode__(self)

    def __repr__(self):
        return slotted__repr__(self)

    def __eq__(self, other):
        if not self.__class__ == other.__class__:
            raise NotImplementedError
        return all(getattr(self, k) == getattr(other, k) for k in self.__slots__)

    def __hash__(self):
        return hash(tuple(getattr(self, k) for k in self.__slots__))


def to_repr(value):
    if isinstance(value, (binary_type, text_type)):
        v = ensure_str(value)
    elif isinstance(value, tuple):
        v = ensure_str(b'({})'.format(b', '.join(to_repr(item) for item in value)))
    else:
        v = repr(value)
    return v


def to_text_type(value):
    if isinstance(value, (binary_type, text_type)):
        v = ensure_text(value)
    elif isinstance(value, tuple):
        v = '({})'.format(', '.join(to_text_type(item) for item in value))
    else:
        v = text_type(value)
    return v


def slotted__unicode__(self):
    v = '{}({})'.format(
        self.__class__.__name__,
        ','.join(
            ':'.join((to_text_type(k), to_text_type(getattr(self, k))))
            for k in self.__slots__
        ),
    )
    return v


def slotted__repr__(self):
    v = b'{}({})'.format(
        self.__class__.__name__,
        b','.join(
            b':'.join((to_repr(k), to_repr(getattr(self, k))))
            for k in self.__slots__
        ),
    )
    return v


def parse_dt(date, time):
    return datetime.strptime('{} {}'.format(date, time), '%d.%m.%y %H:%M')


NO_BAGGAGE = {'НЕТ', 'NO'}


def parse_baggage(baggage):
    # type: (text_type) -> Baggage

    if not baggage:
        return Baggage.from_partner()
    if baggage in NO_BAGGAGE:
        return Baggage.from_partner(included=False)
    if baggage.endswith('КМ'):
        baggage = baggage.replace('КМ', 'PC')
    if baggage.endswith('К'):
        baggage = baggage.replace('К', 'K')
    return BaggageParser(logger=logger).parse_from_string(baggage)
