# -*- coding: utf-8 -*-
import json
import logging
import time
from collections import OrderedDict
from datetime import datetime, timedelta
from itertools import chain

from travel.avia.library.python.ticket_daemon.date import EPOCH


class Timer(object):
    def __init__(self, start_time=None):
        """

        :param datetime start_time: UTC datetime object
        """
        if start_time is not None:
            if isinstance(start_time, datetime):
                self._start_time = (start_time - EPOCH).total_seconds()
            else:
                raise Exception('Unknown start_time format')
        else:
            self._start_time = time.time()

    @property
    def start_time(self):
        return datetime.utcfromtimestamp(self._start_time)

    @property
    def elapsed(self):
        return datetime.utcnow() - self.start_time

    def get_elapsed_seconds(self):
        return time.time() - self._start_time

    def seconds_to_delta(self, delta):
        return delta.total_seconds() - self.get_elapsed_seconds()

    def seconds_to_shift(self, shift):
        return self.seconds_to_delta(timedelta(seconds=shift))


class Timeline(object):
    def __init__(self, logger, defaults=None, start_time=None):
        if isinstance(logger, basestring):
            logger = logging.getLogger(logger)
        self._logger = logger
        self._timer = Timer(start_time=start_time)
        self._id = str(self._timer.start_time.strftime('%d:%H%M%S.%f'))

    def event(self, event_name, extra=None):
        items = self._items(event_name, extra)
        self._send(items)

    def _send(self, items):
        message = self.make_format(items)
        self._logger.info(message)

    def _items(self, event_name, extra):
        items = [
            ('ts', datetime.now().strftime('%Y-%m-%d %H:%M:%S')),
            ('tid', self._id),
            ('elapsed_ms', int(self._timer.get_elapsed_seconds() * 1000)),
            ('event', event_name),
        ]

        return OrderedDict(chain(
            items,  # Чтобы установить значения в начало
            extra or [],
            items  # Чтобы не затёрлись случайно другими значениями
        ))

    def make_format(self, items):
        return '\t'.join(
            '{}={}'.format(k, _somehow_to_str(v)) for k, v in items.items()
        )

    def with_defaults(self, defaults):
        return TimelineWithDefaults(self, defaults)


class TimelineJson(Timeline):
    def __init__(self, logger, defaults=None, start_time=None):
        super(TimelineJson, self).__init__(logger, defaults=defaults, start_time=start_time)

    def make_format(self, items):
        return json.dumps(items)


class TimelineWithDefaults(object):
    def __init__(self, timeline, defaults):
        self._timeline = timeline
        self._defaults = OrderedDict(defaults or []).items()

    def event(self, event_name, extra=None):
        self._timeline.event(
            event_name,
            self._defaults + OrderedDict(extra or []).items()
        )


def _somehow_to_str(value):
    try:
        return str(value)
    except Exception:
        return json.dumps(value, default=repr)
