# -*- coding: utf-8 -*-
import logging
import re
from collections import defaultdict
from datetime import datetime
from itertools import ifilter

from pytils.translit import detranslify, translify
from django.utils.encoding import force_unicode

from travel.avia.library.python.ticket_daemon.memo import memoize

DATETIME_FORMAT = '%Y-%m-%d %H:%M:%S'
RE_DELIMITERS = re.compile(ur'[\s-]+', flags=re.UNICODE)
RE_NORMALIZED_NUMBER = re.compile(
    ur'^([A-ZА-ЯЁ][A-ZА-ЯЁ\d]|[A-ZА-ЯЁ\d][A-ZА-ЯЁ]|[A-ZА-ЯЁ][A-ZА-ЯЁ])\s+0*(\d{1,4}).*$',
    flags=re.UNICODE
)
RE_NORMALIZED_NUMBER_CHARTER = re.compile(
    ur'^([A-ZА-ЯЁ][A-ZА-ЯЁ\d]|[A-ZА-ЯЁ\d][A-ZА-ЯЁ]|[A-ZА-ЯЁ][A-ZА-ЯЁ])\s+0*(\d{1,6}).*$',
    flags=re.UNICODE
)
broken_flight_numbers_log = logging.getLogger('broken_flight_numbers')


class Undefined(object):
    def __getattr__(self, name):
        return self

    def __getitem__(self, name):
        return self

    def __iter__(self):
        return iter([])

    def __nonzero__(self):
        return False

    def __json__(self):
        return None


undefined = Undefined()


def wrap(v):
    if isinstance(v, dict):
        return DictWrapper(v)

    if isinstance(v, list):
        return ListWrapper(v)

    return v


class DictWrapper(dict):
    def __getitem__(self, name):
        try:
            return wrap(dict.__getitem__(self, unicode(name)))

        except KeyError:
            return undefined

    def __getattr__(self, name):
        if name.startswith('__'):
            raise AttributeError

        return self[name]

    def __setattr__(self, name, value):
        self[name] = value


class ListWrapper(list):
    def __getitem__(self, i):
        try:
            return wrap(list.__getitem__(self, i))

        except IndexError:
            return undefined

    def __iter__(self):
        for v in list.__iter__(self):
            yield wrap(v)


def skip_None_values(d):
    if isinstance(d, dict):
        # Type(d) allow to work with dict subclasses and return same type
        return type(d)(
            filter(lambda (key, val): val is not None, d.iteritems())
        )

    return d


def dict_merge(*args):
    result = {}

    for d in args:
        assert isinstance(d, dict)
        result.update(d)

    return result


def group_by(iterable, key):
    groups = defaultdict(list)

    for j in iterable:
        groups[key(j)].append(j)

    return groups


@memoize(lambda flight_number, is_charter: (flight_number, is_charter))
def fix_flight_number(flight_number, is_charter):
    try:
        return _fix_flight_number(flight_number, is_charter)
    except ValueError:
        return None


def _fix_flight_number(flight_number, is_charter):
    original_number = flight_number
    flight_number = force_unicode(flight_number)

    # Переведем все в верхний регистр
    flight_number = flight_number.upper()

    # Удалим пробелы и дефисы
    flight_number = RE_DELIMITERS.sub('', flight_number)

    # Если номер начинается с кода АК, то уберем код из номера
    airline, number = flight_number[:2], flight_number[2:]
    if number.startswith(airline):
        number = number[len(airline):]

    # Если первый символ начинается с цифры, то ничего не делаем
    if not (ord('0') <= ord(airline[0]) <= ord('9')):
        # Переводим все символы в латиницу, если первый символ латинский
        if (
            ord('A') <= ord(airline[0]) <= ord('Z')
        ):
            airline = translify(airline)
        else:
            airline = detranslify(airline)

    flight_number = u'%s %s' % (airline, number)

    # Приведем номер рейса к виду "(Код авиакомпании) (номер, только цифры, не более 4 или 6 для чартеров)"
    normalizer = RE_NORMALIZED_NUMBER_CHARTER if is_charter else RE_NORMALIZED_NUMBER
    match = normalizer.match(flight_number)
    if not match:
        broken_flight_numbers_log.info(u'%s', original_number)
        return None

    return u'%s %s' % match.groups()


def _somehow_to_str(value):
    try:
        return str(value).replace('\t', ' ')
    except Exception:
        return repr(value).replace('\t', ' ')


FALSE_VARIANTS = {'false', '0', 'null', 'none', ''}


def str_to_bool(s):
    if isinstance(s, bool):
        return s
    elif isinstance(s, basestring) and s.lower() in FALSE_VARIANTS:
        return False
    elif s == 0 or s is None:
        return False
    return True


def n_filter(filters, iterable):
    return ifilter(lambda t: all(f(t) for f in filters), iterable)


def parse_datetime_without_seconds(raw_date):
    """
    Example '2018-01-21T13:50:00' for raw_date
    """
    return datetime(
        int(raw_date[0:4]), int(raw_date[5:7]), int(raw_date[8:10]),
        int(raw_date[11:13]), int(raw_date[14:16])
    )


def log_filter(function, iterable, logger):
    for item in iterable:
        if function(item):
            yield item
        else:
            logger.error('Element %r filtered by %r', item, function)
