# -*- coding: utf-8 -*-
from abc import abstractmethod, ABCMeta

from travel.avia.library.python.common.utils import environment

from travel.avia.ticket_daemon.ticket_daemon.lib.yt_loggers.tskv_logger import TskvLogger


class IMinPriceLogger(object):
    __metaclass__ = ABCMeta

    @abstractmethod
    def log(self, query, same_stops_price_importations):
        pass


class MinPriceLogger(IMinPriceLogger):
    """
    Сущность для логгирования минимальных цен
    """
    def __init__(self, logger):
        self._logger = logger

    def log(self, query, same_stops_price_importations):
        """
        :param query: query
        :param same_stops_price_importations: same_stops_price_importations

        :returns Void
        """
        common_data = {
            'passengers': '{adults}_{children}_{infants}'.format(
                adults=query.passengers.get('adults', 0),
                children=query.passengers.get('children', 0),
                infants=query.passengers.get('infants', 0)
            ),
            'direction': '{from_key}_{to_key}'.format(
                from_key=query.point_from.point_key,
                to_key=query.point_to.point_key
            ),
            'date_forward': query.date_forward.strftime('%Y-%m-%d'),
            'date_backward': query.date_backward.strftime('%Y-%m-%d') if query.date_backward else '',
            'klass': query.klass,
            'qid': query.id,
            'national_version': query.national_version,
            'service': query.service
        }

        self._logger.log_many(self._get_record(common_data, info)
                              for info in same_stops_price_importations)

    def _safe_encode(self, container, attr):
        value = None
        if container:
            value = getattr(container, attr)

        value = value or ''

        return value.encode('utf-8')

    def _get_record(self, common_data, info):
        tariff = info.tariff
        national_tariff = info.national_tariff

        forward, backward, partners = self._get_reference(info.variants)
        data = {
            'original_price': tariff.value if tariff else None,
            'original_currency': self._safe_encode(tariff, 'currency'),
            'price': national_tariff.value if national_tariff else None,
            'currency': self._safe_encode(national_tariff, 'currency'),
            'stops': info.stops,
            'forward_routes': ';'.join(forward),
            'backward_routes': ';'.join(backward),
            'partners': ','.join(partners)
        }

        return dict(common_data, **data)

    def _get_reference(self, variants):
        forward = set()
        backward = set()
        partners = set()

        def create_route(segments):
            return ','.join((s.number or '').encode('utf-8') for s in segments)

        for v in variants:
            segments = v.forward.segments
            if segments:
                forward.add(create_route(segments))

            segments = v.backward.segments
            if segments:
                forward.add(create_route(segments))

            partners.add(v.partner.code.encode('utf-8'))

        return forward, backward, partners


min_price_logger = MinPriceLogger(
    logger=TskvLogger(
        name='yt.min_price',
        environment=environment,
        tskv_format='rasp-min-price-log',
    )
)
