# -*- coding: utf-8 -*-
from abc import abstractmethod, ABCMeta

from travel.avia.library.python.common.utils import environment

from travel.avia.ticket_daemon.ticket_daemon.lib.yt_loggers.yt_logger import YtLogger


class IQueryCacheHitLogger(object):
    __metaclass__ = ABCMeta

    @abstractmethod
    def log(self, query, partners, partner_code_to_status, partner_result_metas, ignore_cache, test_id):
        """
            :type query: ticket_daemon.api.query.Query
            :param list partners: partners
            :param dict partner_code_to_status: статусы опроса партнеров по кодам партнеров
            :param dict partner_result_metas: мета-данные результатов, которые сохранены в хранилище.
                Содержат qid, в результате которого в хранилище записались варианты от партнера.
            :param bool ignore_cache: было ли безусловная перезапись кэша при импорте
            :param test_id:
            :return: None
        """


class QueryCacheHitLogger(IQueryCacheHitLogger):
    """
    Сущность для логгирования попадания в cache TD при инициализации поиска
    """
    def __init__(self, logger):
        self._logger = logger

    def log(self, query, partners, partner_code_to_status, partner_result_metas, ignore_cache, test_id):
        self._logger.log_many(self._get_record(
            query=query,
            partner=partner,
            status=partner_code_to_status[partner.code],
            result_meta=partner_result_metas[partner.code],
            ignore_cache=ignore_cache,
            test_id=test_id
        ) for partner in partners)

    def _get_record(self, query, partner, status, result_meta, ignore_cache, test_id):
        status = status or {}

        # Как определяется кто проинициализировал поиск?
        # По наличию статуса мы можем понять, кто-то иницилизировал запрос до нас или нет
        # 1) Если запрос проинициализировали не мы, то информацию о сервисе лежит в статусе
        # 2) Если запрос проинициализировали мы, то информация о сервисе лежит в query
        query_source = status.get('service') or query.service

        return {
            'qid': query.id,
            'init_id': result_meta.get('qid') if result_meta is not None else None,
            'query_source': query_source,
            'partner': partner.code,
            'cache_hit': bool(status),
            'ignore_cache': bool(ignore_cache),
            'from_key': query.point_from.point_key,
            'to_key': query.point_to.point_key,
            'when': str(query.date_forward),
            'return_date': str(query.date_backward or ''),
            'adults': query.adults,
            'children': query.children,
            'infants': query.infants,
            'class': query.klass,
            'service': query.service,
            'national_version': query.national_version,
            'lang': query.lang,
            'experimentsTestIds': test_id or ''
        }


class MultipleQueryCacheHitLogger(IQueryCacheHitLogger):
    def __init__(self, loggers):
        self._loggers = loggers

    def log(self, query, partners, partner_code_to_status, partner_result_metas, ignore_cache, test_id):
        for logger in self._loggers:
            logger.log(query, partners, partner_code_to_status, partner_result_metas, ignore_cache, test_id)


query_cache_hit_logger = MultipleQueryCacheHitLogger(
    loggers=(
        QueryCacheHitLogger(YtLogger(
            name='yt.avia_partner_cache',
            environment=environment
        )),
    )
)
