# -*- coding: utf-8 -*-
from travel.avia.library.python.common.utils import environment

from travel.avia.ticket_daemon.ticket_daemon.lib.yt_loggers.abstract_variant_logger import IVariantsLogger
from travel.avia.ticket_daemon.ticket_daemon.lib.yt_loggers.tskv_logger import TskvLogger


class VariantsLogger(IVariantsLogger):
    """
    Логгирование ответа тикет демона по партнеру
    """
    def __init__(self, logger, environment):
        self._logger = logger
        self._environment = environment

    def log(self, query, partner, partner_variants):
        """
        :param query: query
        :param partner: partner
        :param partner_variants: partner_variants
        :return None
        """
        q = query
        timestamp = self._environment.now_aware()
        point_from = q.point_from
        point_to = q.point_to
        date_forward = q.date_forward.strftime('%Y-%m-%d')
        date_backward = q.date_backward.strftime('%Y-%m-%d') if q.date_backward else None

        common_data = {
            'timestamp': timestamp.strftime('%Y-%m-%d %H:%M:%S'),
            'timezone': timestamp.strftime('%z'),
            'partner': partner.code.encode('utf-8'),
            'type': 'plane',
            'date_forward': date_forward,
            'date_backward': date_backward,
            'object_from_type': point_from.__class__.__name__,
            'object_from_id': point_from.id,
            'object_from_title': (point_from.title or '').encode('utf-8'),
            'object_to_type': point_to.__class__.__name__,
            'object_to_id': point_to.id,
            'object_to_title': (point_to.title or '').encode('utf-8'),
            'class_{}_seats'.format(q.klass): q.passengers_count,
            'adults': q.passengers.get('adults', 0),
            'children': q.passengers.get('children', 0),
            'infants': q.passengers.get('infants', 0),
            'national_version': (getattr(q, 'national_version', None) or '').encode('utf-8'),
            'qid': getattr(q, 'id', None),
        }

        self._logger.log_many(
            dict(self._get_record(v), **common_data)
            for v in partner_variants
            if len(v.forward.segments) == 1 and len(v.backward.segments) <= 1
        )

    def _get_record(self, variant):
        """
        :param variant: variant
        :return: Данные специфичные для каждого варианта
        """
        v = variant
        tariff = '%s %s' % (
            getattr(v.tariff, 'value', None),
            getattr(v.tariff, 'currency', None)
        )

        return {
            'route_uid': ';'.join((s.number or '').encode('utf-8') for s in v.all_segments),
            'baggages': ';'.join(str(s.baggage).encode('utf-8') if s.baggage else '' for s in v.all_segments),
            'class_{}_price'.format(v.klass): tariff,
            'forward_stops': len(v.forward.segments) - 1,
            'backward_stops': (len(v.backward.segments) - 1) if v.backward else 0,
            'raw_tariffs': '{}'  # deprecated field
        }


variants_logger = VariantsLogger(
    logger=TskvLogger(
        name='rasp.order.dump_seat_price',
        environment=environment,
        tskv_format='rasp-tariffs-log',
    ),
    environment=environment,
)
