# -*- coding: utf-8 -*-
import logging
from textwrap import dedent

from django.core.management import BaseCommand
from marshmallow import fields, Schema, post_load
from marshmallow.fields import Date, Integer, Constant, String, validate

from travel.avia.library.python.common.models.geo import Point, Settlement, Station
from travel.avia.library.python.common.models.partner import DohopVendor
from travel.avia.library.python.common.utils import environment

from travel.avia.ticket_daemon.ticket_daemon.api.flights import FlightFabric
from travel.avia.ticket_daemon.ticket_daemon.api.query import Query, QueryAllException
from travel.avia.ticket_daemon.ticket_daemon.daemon.big_beauty_collector import BigBeautyCollector
from travel.avia.ticket_daemon.ticket_daemon.daemon.importer import ModuleImporter, Importer
from travel.avia.ticket_daemon.ticket_daemon.lib.partner_secret_storage import partner_secret_storage
from travel.avia.ticket_daemon.ticket_daemon.lib.timer import Timeline
from travel.avia.ticket_daemon.ticket_daemon.lib.yt_loggers.yt_logger import YtLogger
from travel.avia.ticket_daemon.ticket_daemon.lib.yt_loggers.wizard_search_result_logger import WizardSearchResultLogger
from travel.avia.ticket_daemon.ticket_daemon.management.commands._loggers import setup_logger
from travel.proto.avia.wizard.search_result_pb2 import SearchResult

log = logging.getLogger(__name__)


class Command(BaseCommand):
    help = dedent("""\
    Make request to query_partners daemon http-method.
    Example usage:
    common/virtualenv/bin/python -W ignore ./manage.py \
        run_partner_module -f mow -t svx -s ticket -d 2017-03-03 -b 2017-03-12 -v3 -m ozon -C
    """).strip()

    def add_arguments(self, parser):
        parser.add_argument('-f', '--point_from', dest='point_from')
        parser.add_argument('-t', '--point_to', dest='point_to')
        parser.add_argument('-n', '--national', dest='national')
        parser.add_argument('-d', '--date_forward', dest='date_forward')
        parser.add_argument('-b', '--date_backward', dest='date_backward')
        parser.add_argument('-s', '--service', dest='service')
        parser.add_argument('-m', '--module', dest='module')
        parser.add_argument(
            '-C', '--use-cache', action='store_true', default=False,
            dest='use_cache', help='Use cached response',
        )
        parser.add_argument(
            '-q', '--quiet', action='store_true', default=False,
            dest='do_not_add_stdout_handler', help='Do not add stdout handler',
        )

    def handle(self, **options):
        setup_logger(log, options['verbosity'],
                     not options.get('do_not_add_stdout_handler'))
        try:
            query = self.make_query(options)
            _run_query(query, options['module'], options['use_cache'])
        except QueryAllException as e:
            log.error(e)
        except Exception as e:
            log.exception('Handle fail: %r', e)

    def make_query(self, options):
        options['service'] = options['service']
        options = {k: v for k, v in options.iteritems() if v is not None}
        query_params = QuerySchema(strict=True).load(options).data
        log.debug('Query params: %r', query_params)
        return Query.from_qid(Query(**query_params).id)


class Partner(object):
    def __init__(self, code):
        self.code = code


def _run_query(q, importer_code, use_cache):
    log.info('Query: %s', q.qid_msg)

    q.timeline = Timeline(log)
    q.prepare_attrs_for_import()
    q.trackers = {}
    q.need_store_tracks = True
    q.tracker_use_cache = use_cache

    partner_secret_storage.recache()
    flight_fabric = FlightFabric()

    if importer_code == 'dohop':
        partners = list(DohopVendor.objects.all())
    else:
        partners = [Partner(importer_code)]
    wizard_search_result_logger = WizardSearchResultLogger(
        logger=YtLogger(name=__name__, environment=environment)
    )
    big_beauty_collector = BigBeautyCollector(q, SearchResult(), wizard_search_result_logger)

    log.info('Importer code: %s', importer_code)
    importer_cls = Importer.get_importer_cls_by_code(importer_code)
    log.info('Importer cls: %s', importer_cls.__name__)
    importer = importer_cls(
        code=importer_code,
        partners=partners,
        query=q,
        response_collector=None,
        flight_fabric=flight_fabric,
        big_beauty_collector=big_beauty_collector,
    )
    log.info('Query module: {}'.format(
        ModuleImporter.import_module('travel.avia.ticket_daemon.ticket_daemon.partners.%s' % importer_code)),
    )
    q = importer.q

    chunked_queryfun = importer.prepare_chunked_queryfun(importer_code, q)

    variants = []
    pagenum = 0
    for pagenum, chunk in enumerate(chunked_queryfun(q), start=1):
        vs = list(chunk)
        log.info('Chunk %s size: %s', pagenum, len(vs))
        variants.extend(vs)

    log.info('Total pages: %s, variants: %s', pagenum, len(variants))
    t = q.trackers.values()[0]
    log.info('Tracker: %s', t.key)
    log.info('Query time: %s', t.query_time)

    # for j, v in enumerate(variants, start=1):
    #     logger.info('%s. %s', j, show_variant(v))


def show_variant(v):
    return 'F: {} .. B: {}'.format(
        ' | '.join(show_flight(s) for s in v.forward.segments),
        ' | '.join(show_flight(s) for s in v.backward.segments),
    )


def show_flight(f):
    return '{}-{} [{}]'.format(
        f.station_from_iata, f.station_to_iata, f.number
    )


def get_point(value):
    try:
        return Point.get_by_key(value)
    except Exception:
        pass

    value = value.lower()

    try:
        return Settlement.objects.get(iata__iexact=value)
    except Exception:
        pass

    try:
        return Settlement.objects.get(sirena_id__iexact=value)
    except Exception:
        pass

    try:
        return Station.objects.filter(code_set__system__code='iata'). \
            distinct().get(code_set__code__iexact=value)
    except Exception:
        pass

    try:
        return Station.objects.filter(code_set__system__code='sirena'). \
            distinct().get(code_set__code__iexact=value)
    except Exception:
        pass


class QuerySchema(Schema):
    national_version = String(load_from='national', missing='ru')
    service = String(required=True)

    point_from = fields.Function(deserialize=get_point)
    point_to = fields.Function(deserialize=get_point)

    date_forward = Date()
    date_backward = Date(required=False, missing=None)
    klass = String(missing='economy', validate=validate.OneOf(['economy', 'business']))

    t_code = Constant('plane')
    lang = Constant('ru')

    adults = Integer(missing=1)
    children = Integer(missing=0)
    infants = Integer(missing=0)

    @post_load
    def compose_passengers(self, item):
        item['passengers'] = {
            'adults': item.pop('adults'),
            'children': item.pop('children'),
            'infants': item.pop('infants'),
        }
        return item
