# -*- coding: utf-8 -*-

from django.db import models
from django.db.models import fields
from django.utils.translation import ugettext_lazy as _

from travel.avia.library.python.common.models.geo import StationCode, Station, Settlement
from travel.avia.library.python.common.models.partner import Partner


class QueryBlackList(models.Model):
    CLASS_CHOICES = [
        (u'economy', _(u'Эконом')),
        (u'business', _(u'Бизнес')),
        (u'first', _(u'Первый')),
    ]

    partner = models.ForeignKey(
        Partner, null=True, blank=True, verbose_name=_(u"партнер")
    )

    country_from = models.ForeignKey(
        'www.Country', null=True, blank=True,
        related_name=u"query_black_list_country_from",
        verbose_name=_(u"страна отправления")
    )

    settlement_from = models.ForeignKey(
        'www.Settlement', null=True, blank=True,
        related_name=u"query_black_list_settlement_from",
        verbose_name=_(u"город отправления")
    )

    station_from = models.ForeignKey(
        'www.Station', null=True, blank=True,
        related_name=u"query_black_list_station_from",
        verbose_name=_(u"станция отправления")
    )

    country_to = models.ForeignKey(
        'www.Country', null=True, blank=True,
        related_name=u"query_black_list_country_to",
        verbose_name=_(u"страна прибытия")
    )

    settlement_to = models.ForeignKey(
        'www.Settlement', null=True, blank=True,
        related_name=u"query_black_list_settlement_to",
        verbose_name=_(u"город прибытия")
    )

    station_to = models.ForeignKey(
        'www.Station', null=True, blank=True,
        related_name=u"query_black_list_station_to",
        verbose_name=_(u"станция прибытия")
    )

    company = models.ForeignKey(
        'www.Company', null=True, blank=True,
        related_name=u"query_black_list_company",
        verbose_name=_(u"Перевозчик")
    )

    flight_number = fields.CharField(
        _(u'Номер рейса'), null=True, blank=True, max_length=10, db_index=True
    )

    when_from = fields.DateField(
        _(u'Дата рейса от'), null=True, blank=True, db_index=True
    )

    when_to = fields.DateField(
        _(u'Дата рейса до'), null=True, blank=True, db_index=True
    )

    price_from = models.FloatField(_(u'Цена от'), blank=True, null=True)

    price_to = models.FloatField(_(u'Цена до'), blank=True, null=True)

    currency = models.ForeignKey(
        'currency.Currency', null=True, blank=True,
        related_name=u"query_black_list_price_currency",
        verbose_name=_(u"Валюта")
    )

    klass = models.CharField(
        _(u'Класс'), choices=CLASS_CHOICES, null=True, blank=True, max_length=10
    )

    description = fields.CharField(
        _(u'Причина бана'), null=False, blank=False, max_length=255
    )

    active_to = fields.DateField(
        _(u'Активно до'), null=False, blank=False, db_index=True
    )

    active = fields.BooleanField(
        _(u'Активно'), null=False, blank=False, db_index=True
    )

    national_version = models.CharField(
        _(u'Нац. версия'), blank=True, null=True, max_length=3
    )

    allow = fields.BooleanField(
        _(u'Разрешить'), null=False, blank=False, default=False
    )

    def __unicode__(self):
        return _(u"Правило № %s") % self.id

    class Meta:
        app_label = 'order'
        verbose_name = _(u"правило черного списка")
        verbose_name_plural = _(u"правила черного списка")


class ActualDirection(models.Model):
    """
    Данная модель используется для фильтрации запросов к партнерам по направлениям.
    Позволяет запрашивать партнера только на те направления, где у него есть рейсы.
    Заполняется при импорте.
    """

    partner = models.ForeignKey(Partner, null=False, blank=False, verbose_name=_(u"партнер"))
    station_from = models.ForeignKey(
        'www.Station', null=False, blank=False,
        verbose_name=_(u'начальная станция'),
        related_name='station_can_ask_from'
    )
    station_to = models.ForeignKey(
        'www.Station', null=False, blank=False,
        verbose_name=_(u'конечная станция'),
        related_name='station_can_ask_to'
    )
    settlement_from = models.ForeignKey(
        'www.Settlement', null=True, blank=True, default=None,
        verbose_name=_(u'начальный город'),
        related_name='settlement_can_ask_from'
    )
    settlement_to = models.ForeignKey(
        'www.Settlement', null=True, blank=True, default=None,
        verbose_name=_(u'конечный город'),
        related_name='settlement_can_ask_to'
    )

    @classmethod
    def get_code_pairs(cls, partner_id, codesystem_id, point_from, point_to):
        code_pairs = []

        query = cls.objects.filter(partner_id=partner_id)

        if isinstance(point_from, Station):
            query = query.filter(station_from_id=point_from.id)

        elif isinstance(point_from, Settlement):
            query = query.filter(settlement_from_id=point_from.id)

        else:
            return code_pairs

        if isinstance(point_to, Station):
            query = query.filter(station_to_id=point_to.id)

        elif isinstance(point_to, Settlement):
            query = query.filter(settlement_to_id=point_to.id)

        else:
            return code_pairs

        # ВАЖНО! При изменении проверьте, что запрос не содежит JOIN
        directions = list(query.values('station_from_id', 'station_to_id').distinct())

        if not directions:
            return code_pairs

        station_ids = set()

        for d in directions:
            station_ids.add(d['station_from_id'])
            station_ids.add(d['station_to_id'])

        codes_by_station_ids = StationCode.StationCodeByStationIdGetter(
            codesystem_id, station_ids)

        for d in directions:
            code_from = codes_by_station_ids.get(d['station_from_id'])
            code_to = codes_by_station_ids.get(d['station_to_id'])

            if code_from and code_to:
                code_pairs.append((code_from, code_to))

        return code_pairs

    class Meta:
        app_label = 'order'
