# -*- coding: utf-8 -*-
import ujson
import urllib
from collections import OrderedDict
from datetime import datetime
from logging import getLogger

import requests
from decorator import decorator

from travel.avia.ticket_daemon.ticket_daemon.api.flights import Variant
from travel.avia.ticket_daemon.ticket_daemon.daemon.utils import sleep_every
from travel.avia.ticket_daemon.ticket_daemon.lib.baggage import BaggageParser
from travel.avia.ticket_daemon.ticket_daemon.lib.currency import Price
from travel.avia.ticket_daemon.ticket_daemon.lib.decorators import pipe
from travel.avia.ticket_daemon.ticket_daemon.lib.http import url_complement_missing
from travel.avia.ticket_daemon.ticket_daemon.lib.tracker import QueryTracker
from travel.avia.ticket_daemon.ticket_daemon.lib.partner_secret_storage import partner_secret_storage
from travel.avia.ticket_daemon.ticket_daemon.lib.utils import skip_None_values

log = getLogger(__name__)
baggage_parser = BaggageParser(logger=log)

API_URL = 'https://flightapi.linkwi.se/aegeanairlines/search.php'
LOGIN = 'yandex'
PASSWORD = partner_secret_storage.get(
    importer_name='aegean', namespace='PASSWORD'
)
AEGEANAIR_DEEPLINK_BASE_URL = 'https://en.aegeanair.com/PostHandler.axd'
LANG_MAP = {
    'ru': 9,
    'ua': 9,
    'kz': 9,
    'com.tr': 3,
    'tr': 3,
    'com': 3,
}
CULTURES_MAP = {
    'ru': 'RU',
    'ua': 'UA',
    'kz': 'KZ',
    'tr': 'TR',
    'com': 'US',
    'com.tr': 'TR',
}


@QueryTracker.init_query
def query(tracker, q):
    raw_data = get_data(tracker, q)
    variants = parse_response(raw_data, q)

    return variants


def get_data(tracker, q):
    r = tracker.wrap_request(
        requests.get,
        API_URL,
        auth=(LOGIN, PASSWORD),
        params={
            'from': q.iata_from.encode('utf-8'),
            'to': q.iata_to.encode('utf-8'),
            'date': q.date_forward.strftime('%d/%m/%Y'),
            'date_return': q.date_backward and q.date_backward.strftime('%d/%m/%Y') or None,
            'adults': q.passengers.get('adults', 1),
            'children': q.passengers.get('children', 0),
            'infants': q.passengers.get('infants', 0),
            'format': 'json',
        },
    )

    data = ujson.loads(r.content)
    return data


def parse_response(data, q):
    variants = []

    if 'error' in data:
        log.error('Api response error code: %s, message: %s', data.get('code'), data.get('description'))
        return variants

    if not data['response']['results']:
        return variants

    for variant_description in sleep_every(data['response']['results']):
        v = Variant()

        v.forward.segments = parse_segments(
            segments=variant_description['outbound_flights'][0]['legs'],
            flight_fabric=q.importer.flight_fabric,
        )

        if variant_description['return_flights']:
            v.backward.segments = parse_segments(
                segments=variant_description['return_flights'][0]['legs'],
                flight_fabric=q.importer.flight_fabric,
            )

        v.tariff = Price(
            float(variant_description['prices'][0]['totalamount']),
            currency=variant_description['prices'][0]['currency'],
        )

        v.klass = q.klass
        v.url = create_deeplink(v, q, variant_description)
        v.order_data = {'url': v.url}

        variants.append(v)

    return variants


@pipe(list)
def parse_segments(segments, flight_fabric):
    for segment in segments:
        yield flight_fabric.create(
            company_iata=segment['airline']['code'],
            pure_number=segment['flightnumber'],
            station_from_iata=segment['departure']['airport'],
            station_to_iata=segment['arrival']['airport'],
            local_departure=datetime.strptime(
                segment['departure']['date'],
                '%Y-%m-%dT%H:%M',
            ),
            local_arrival=datetime.strptime(
                segment['arrival']['date'],
                '%Y-%m-%dT%H:%M',
            ),
        )


def get_deeplink_cabin_class(str_class):
    str_class = str_class.lower()
    if 'business' in str_class:
        return 'BUSINESS'
    elif 'flex' in str_class:
        return 'FLEX'
    return 'ECONOMY'


def get_deeplink_flight_number(flight):
    if flight.company_iata:
        return flight.number[3:]
    return flight.number[1:]


@decorator
def tracking_link(func, *args, **kw):
    return 'http://go.linkwi.se/z/{PROGRAM_ID}/{AFFILIATE_ID}/?linkurl={link_url}'.format(
        PROGRAM_ID='73-0',
        AFFILIATE_ID='CD23355',
        link_url=urllib.quote(func(*args, **kw))
    )


@tracking_link
def create_deeplink(v, q, raw_variants):
    params = skip_None_values(OrderedDict((
        ('Referer', 'eticket'),
        ('lang', LANG_MAP[q.national_version]),
        ('AirportFrom', v.forward.segments[0].station_from_iata),
        ('AirportTo', v.forward.segments[-1].station_to_iata),
        ('TravelType', 'R' if q.date_backward else 'O'),
        ('directFlights', 'indirect'),
        ('DateDeparture', v.forward.segments[0].local_departure.strftime('%d/%m/%Y')),
        ('DateReturn', v.backward.segments[0].local_departure.strftime('%d/%m/%Y') if v.backward.segments else None),
        ('Btn_DaysRange', 3),
        ('AdultsNum', q.passengers.get('adults', 1)),
        ('Children12Nums', q.passengers.get('children', 0)),
        ('Children5Num', 0),
        ('InfantsNum', q.passengers.get('infants', 0)),
        ('bounds[0].fareFamily', get_deeplink_cabin_class(raw_variants['prices'][0]['class'])),
        ('bounds[0].flights[0].marketingCarrier', raw_variants['outbound_flights'][0]['legs'][0]['airline']['code']),
        ('bounds[0].flights[0].flightNumber', get_deeplink_flight_number(v.forward.segments[0])),
        ('bounds[1].fareFamily', get_deeplink_cabin_class(raw_variants['prices'][0]['class'])),
        ('bounds[1].flights[0].marketingCarrier', (raw_variants['return_flights'][0]['legs'][0]['airline']['code'] if raw_variants['return_flights'] else None)),
        ('bounds[1].flights[0].flightNumber', (get_deeplink_flight_number(v.backward.segments[0]) if v.backward.segments else None)),
        ('origin', v.forward.segments[0].station_from_iata),
        ('destination', v.forward.segments[-1].station_to_iata),
        ('country', CULTURES_MAP[q.national_version]),
    )))

    return url_complement_missing(
        AEGEANAIR_DEEPLINK_BASE_URL,
        params,
    )
