# -*- coding: utf-8 -*-
from collections import defaultdict
from datetime import datetime
from logging import getLogger
from typing import Iterable, Dict, List

import gevent
import requests

from travel.avia.library.python.avia_data.models.amadeus_merchant import AmadeusMerchant
from travel.avia.ticket_daemon.ticket_daemon.api.flights import Variant, Segment, FlightFabric, min_class
from travel.avia.ticket_daemon.ticket_daemon.lib.currency import Price
from travel.avia.ticket_daemon.ticket_daemon.lib.http import update_query_string
from travel.avia.ticket_daemon.ticket_daemon.api.query import QueryIsNotValid, Query
from travel.avia.ticket_daemon.ticket_daemon.daemon.utils import sleep_every, BadPartnerResponse
from travel.avia.ticket_daemon.ticket_daemon.lib.tracker import QueryTracker
from travel.avia.ticket_daemon.ticket_daemon.lib.partner_secret_storage import partner_secret_storage

log = getLogger(__name__)

AMADEUS_API_URL = 'https://api.connect.travelaudience.com/api/search'
AMADEUS_PASSWORD = partner_secret_storage.get(
    importer_name='amadeus', namespace='PASSWORD'
)
AMADEUS_MAX_NUMBER_OF_RESULTS_PER_MERCHANT = 1000

CURRENCIES_MAP = {'ru': 'RUB', 'ua': 'UAH', 'tr': 'TRY', 'com': 'EUR', 'kz': 'KZT'}
CULTURES_MAP = {'ru': 'ru-RU', 'ua': 'ru-RU', 'tr': 'tr-TR', 'com': 'en-EN'}
KLASSES_MAP = {'economy': 'ECONOMY', 'business': 'BUSINESS', 'first': 'FIRST'}
KLASS_REVERSE_MAP = {'ECONOMY': 'economy', 'PREMIUM_ECONOMY': 'economy', 'BUSINESS': 'business', 'FIRST': 'first'}


def validate_query(q):
    if not (hasattr(q, 'iata_real_to') and hasattr(q, 'iata_real_from')):
        raise QueryIsNotValid('Only IATA codes allowed: %s %s' % (
            q.iata_from.encode('utf-8'),
            q.iata_to.encode('utf-8')
        ))


@QueryTracker.init_query
def query(tracker, q):
    bad_responses = []
    empty = True
    for g in gevent.iwait([
        gevent.spawn(get_data, tracker, params)
        for params in build_search_params(q)
    ]):
        if isinstance(g.value, BadPartnerResponse):
            bad_responses.append(g.value)
            continue

        if g.value is not None:
            empty = False
            yield list(parse_response(g.value, q))

    if empty and bad_responses:
        raise bad_responses[0]


def get_data(tracker, params):
    url = update_query_string(AMADEUS_API_URL, params)
    r = tracker.wrap_request(requests.get, url)
    return r.json()


def parse_flight(flight, flight_fabric):
    # type: (dict, FlightFabric) -> Segment
    return flight_fabric.create(
        station_from_iata=flight['origin']['airport'],
        station_to_iata=flight['destination']['airport'],
        local_departure=parse_datetime(flight['departs_at']),
        local_arrival=parse_datetime(flight['arrives_at']),
        company_iata=flight['marketing_airline'],
        pure_number=flight['flight_number'],
        klass=KLASS_REVERSE_MAP.get(flight['booking_info']['travel_class'])
    )


def parse_response(json, q):
    for variant in sleep_every(json.get('results', [])):
        try:
            if KLASS_REVERSE_MAP.get(variant.get('travel_class')) != q.klass:
                continue
            v = Variant()
            v.klass = q.klass

            v.forward.segments = [parse_flight(flight, q.importer.flight_fabric)
                                  for flight in variant['outbound']['flights']]

            if q.date_backward:
                v.backward.segments = [parse_flight(flight, q.importer.flight_fabric)
                                       for flight in variant['inbound']['flights']]

            v.tariff = Price(
                float(variant.get('fare').get('total_price')),
                variant.get('fare').get('currency')
            )
            v.url = variant['deep_link']
            v.order_data = {'url': v.url}

            v.amadeus_merchant_id = variant.get('merchant').lower()
            v.klass = min_class(v.klasses)

            yield v

        except Exception as e:
            log.error('Error while parsing variant: %r', e)


def build_search_params(q):
    # type: (Query) -> Iterable[Dict[str, str]]

    all_partners = [partner for partner in q.importer.partners if isinstance(partner, AmadeusMerchant)]  # type: List[AmadeusMerchant]
    partners_by_locale_and_currency = defaultdict(list)

    default_locale = CULTURES_MAP.get(q.national_version, 'ru-RU')
    default_currency = CURRENCIES_MAP.get(q.national_version, 'RUB')

    for partner in all_partners:
        partners_by_locale_and_currency[(
            partner.get_language_by_national_version(q.national_version, default_locale),
            partner.get_currency_by_national_version(q.national_version, default_currency),
        )].append(partner)

    for (locale, currency), partners in partners_by_locale_and_currency.items():
        params = {
            'origin': q.iata_from,
            'destination': q.iata_to,
            'locale': locale,
            'currency': currency,
            'departure_date': q.date_forward.strftime('%Y-%m-%d'),
            'affiliate_key': AMADEUS_PASSWORD,
            'adults': q.passengers.get('adults', 0),
            'children': q.passengers.get('children', 0),
            'infants': q.passengers.get('infants', 0),
            'class': KLASSES_MAP[q.klass],
            'number_of_results': AMADEUS_MAX_NUMBER_OF_RESULTS_PER_MERCHANT,
            'include_merchants': ','.join(
                [p.merchant_id.upper() for p in partners]
            )
        }

        if q.date_backward:
            params.update({
                'return_date': q.date_backward.strftime('%Y-%m-%d'),
            })

        yield params


def parse_datetime(raw_datetime):
    return datetime.strptime(raw_datetime, '%Y-%m-%dT%H:%M')
