# -*- coding: utf-8 -*-
import json
import logging

import requests

from travel.avia.ticket_daemon.ticket_daemon.api.flights import Variant
from travel.avia.ticket_daemon.ticket_daemon.daemon.utils import sleep_every
from travel.avia.ticket_daemon.ticket_daemon.lib.baggage import Baggage
from travel.avia.ticket_daemon.ticket_daemon.lib.currency import Price
from travel.avia.ticket_daemon.ticket_daemon.lib.tracker import QueryTracker
from travel.avia.ticket_daemon.ticket_daemon.lib.utils import parse_datetime_without_seconds
from travel.avia.ticket_daemon.ticket_daemon.api.query import Query
from travel.avia.ticket_daemon.ticket_daemon.lib.partner_secret_storage import partner_secret_storage


log = logging.getLogger(__name__)

KLASS_NAME2CODE = {'economy': 'E', 'business': 'D'}
KLASS_CODE2NAME = {'E': 'economy', 'B': 'business'}
PARTNER = partner_secret_storage.get(
    importer_name='anywayanyday',
    namespace='PARTNER'
)


def validate_query(q):
    q.validate_klass(KLASS_NAME2CODE)


@QueryTracker.init_query
def query(tracker, q):
    return parse_response(get_data(tracker, q), q)


def get_data(tracker, q):
    r = tracker.wrap_request(
        requests.get,
        'http://api.anywayanyday.com/YandexSearch/',
        params=build_query_params(q)
    )
    return json.loads(r.content)


def build_query_params(q):
    # type: (Query) -> dict
    params = {
        'departure': q.iata_from,
        'arrival': q.iata_to,
        'date_forward': q.date_forward.strftime('%Y-%m-%d'),
        'adults': q.passengers.get('adults', 0),
        'children': q.passengers.get('children', 0),
        'infants': q.passengers.get('infants', 0),
        'class': KLASS_NAME2CODE[q.klass],
        'locale': q.lang,
        'partner': PARTNER,
    }

    if q.date_backward:
        params['date_backward'] = q.date_backward.strftime('%Y-%m-%d')

    return params


def parse_response(data, q):
    variants = []
    for raw_variant in sleep_every(data['variant']):
        try:
            v = parse_variant(q, raw_variant)
            if v:
                variants.append(v)
        except Exception:
            log.exception('Error in parse_variant')
    return variants


def get_baggage(raw):
    try:
        is_baggage_raw = raw.get('luggage')
        is_baggage = is_baggage_raw == 'true'
        if is_baggage_raw is None:
            return Baggage.from_partner()
        elif is_baggage is False:
            return Baggage.from_partner(0)
        elif is_baggage is True:
            pieces = int(raw.get('pieces_of_luggage', 0))
            weight = int(raw.get('luggage_weight', 0))
            if pieces > 0 and weight == 0:
                return Baggage.from_partner(pieces=pieces)
            return Baggage.from_partner(weight=weight, pieces=pieces)
        else:
            log.error('Invalid "is_baggage": %s', is_baggage)
    except Exception, exc:
        log.error('Baggage parsing exception: %s on element %s', exc, raw)

    return Baggage.from_partner()


def parse_variant(q, raw):
    v = Variant()
    v.klass = q.klass

    v.forward.segments = parse_flights(raw['route_forward'], q.importer.flight_fabric)
    if q.date_backward and 'route_backward' not in raw:
        log.error('No route_backward in the response for return-trip query')
        return
    if q.date_backward:
        v.backward.segments = parse_flights(raw['route_backward'], q.importer.flight_fabric)

    segments_klasses = list(set(s.klass for s in v.all_segments))
    if len(segments_klasses) != 1 or segments_klasses[0] != q.klass:
        log.debug('Service classes does not match: %r != %r', segments_klasses, q.klass)
        return

    currency = 'RUR' if raw['currency'] == 'RUB' else raw['currency']
    v.tariff = Price(float(raw['price']), currency)
    v.url = raw['url']
    v.order_data = {'url': v.url}

    return v


def parse_flights(raw_flights, flight_fabric):
    return [
        flight_fabric.create(
            company_iata=raw['company_code'],
            pure_number=raw['route_code'],
            fare_code=raw.get('farecode'),
            station_from_iata=raw['departure_airport_code'],
            station_to_iata=raw['arrival_airport_code'],
            local_departure=parse_datetime_without_seconds(raw['departure_datetime']),
            local_arrival=parse_datetime_without_seconds(raw['arrival_datetime']),
            klass=KLASS_CODE2NAME[raw['fare']['class']],
            baggage=get_baggage(raw['fare']),
            charter=raw['fare'].get('charter') == 'true',
            selfconnect=raw['fare'].get('selfconnect') == 'true',
        )
        for raw in raw_flights
    ]
