# -*- coding: utf-8 -*-

import hashlib
from datetime import datetime, timedelta
from logging import getLogger

import requests
from django.template import loader
from lxml import etree

from travel.avia.ticket_daemon.ticket_daemon.api.flights import IATAFlight, Variant
from travel.avia.ticket_daemon.ticket_daemon.daemon.utils import sleep_every
from travel.avia.ticket_daemon.ticket_daemon.lib.currency import Price
from travel.avia.ticket_daemon.ticket_daemon.lib.tracker import QueryTracker

log = getLogger(__name__)

AVIA_CASSA_URL = 'https://search.biletix.ru/bitrix/components/travelshop/ibe.soap/travelshop_booking.php'
AVIA_CASSA_LOGIN = '[partner]||aviacassa.yandex.ru'
AVIA_CASSA_PASSWD = 'stAwRasp3Ded'
AVIA_CASSA_SHARED_SECRET = 'kUraw22axage'

KLASS_MAP = {
    'economy': u'E',
    'business': u'B',
    'first': u'F',
}


@QueryTracker.init_query
def query(tracker, q):
    xml = get_data(tracker, q)

    variants = parse_response(xml, q.klass)

    return variants


def get_data(tracker, q):
    params = [
        ('session_token', AVIA_CASSA_LOGIN + ':' + AVIA_CASSA_PASSWD + ':'),
        ('owrt', 'RT' if q.date_backward else 'OW'),
        ('departure_point', q.iata_from),
        ('arrival_point', q.iata_to),
        ('outbound_date', q.date_forward.strftime('%d.%m.%Y')),
    ]

    if q.date_backward:
        params += [('return_date', q.date_backward.strftime('%d.%m.%Y')), ]

    params += [
        ('adult_count', q.passengers.get('adults', 0)),
        ('child_count', q.passengers.get('children', 0)),
        ('infant_count', q.passengers.get('infants', 0)),
        ('class', KLASS_MAP.get(q.klass, 'E')),
    ]

    conc = u''.join([unicode(v) for n, v in params]) + AVIA_CASSA_SHARED_SECRET

    params.append(('hash', hashlib.md5(conc.encode('utf-8')).hexdigest()))

    query_xml = loader.render_to_string(
        'partners/biletix.xml', dict(params)
    ).strip()

    r = tracker.wrap_request(
        requests.post,
        AVIA_CASSA_URL,
        headers={
            'Content-Type': 'text/xml; charset=utf-8',
            'SOAPAction': 'urn:#GetOptimalFares',
        },
        data=query_xml.encode('utf-8')
    )

    return r.content


NSMAP = {'ns1': 'http://www.tais.ru/'}


def parse_response(xml, klass):
    tree = etree.fromstring(xml)
    variants = []

    for flight_tag in sleep_every(tree.xpath('//ns1:GetOptimalFaresOffer', namespaces=NSMAP)):
        try:
            charter = flight_tag.xpath(
                'ns1:charter', namespaces=NSMAP)[0].text == "Y"

        except IndexError:
            charter = False

        tariff = float(
            flight_tag.xpath('ns1:total_price', namespaces=NSMAP)[0].text)

        direction_to = flight_tag.xpath(
            'ns1:directions/ns1:GetOptimalFaresDirection[ns1:direction="TO"]',
            namespaces=NSMAP
        )

        flights_to = direction_to and direction_to[0].xpath(
            'ns1:flights/ns1:GetOptimalFaresFlight', namespaces=NSMAP
        ) or []

        if not flights_to:
            continue

        flights_to = [{'xml': f} for f in flights_to]

        # Собрать сегменты для каждого полёта туда
        for flight_to in flights_to:
            flight_to['seg'] = parse_flights(
                flight_to['xml'].xpath('ns1:segments/ns1:AirSegment',
                                       namespaces=NSMAP)
            )

        direction_back = flight_tag.xpath(
            'ns1:directions/ns1:GetOptimalFaresDirection[ns1:direction="BACK"]',
            namespaces=NSMAP
        )
        flights_back = direction_back and direction_back[0].xpath(
            'ns1:flights/ns1:GetOptimalFaresFlight', namespaces=NSMAP
        ) or []
        flights_back = [{'xml': f} for f in flights_back]

        # Собрать сегменты для каждого полёта обратно
        for flight_back in flights_back:
            flight_back['seg'] = parse_flights(
                flight_back['xml'].xpath(
                    'ns1:segments/ns1:AirSegment', namespaces=NSMAP)
            )

        flights_back = filter(lambda f: bool(f['seg']), flights_back) or [None]

        baseurl = flight_tag.xpath('ns1:link', namespaces=NSMAP)[0].text

        for flight_to in flights_to:
            for flight_back in flights_back:
                v = Variant()
                v.forward.segments = flight_to['seg']
                if flight_back:
                    v.backward.segments = flight_back['seg']

                    # RASPTICKETS-388
                    if bad_cross_segments(v):
                        continue

                v.tariff = Price(tariff)

                v.url = baseurl + flight_to['xml'].xpath('ns1:link', namespaces=NSMAP)[0].text

                if flight_back:
                    v.url += flight_back['xml'].xpath('ns1:link', namespaces=NSMAP)[0].text

                v.charter = charter

                v.klass = klass
                v.order_data = {'url': v.url, }

                variants.append(v)

    return variants


def bad_cross_segments(v):
    return (
        v.backward.segments and
        v.forward.segments[-1].local_arrival and
        v.backward.segments[0].local_departure and (
            v.forward.segments[-1].local_arrival + timedelta(hours=1) >
            v.backward.segments[0].local_departure
        )
    )


def parse_flights(segments):
    flights = []

    for segment in segments:
        f = IATAFlight()

        f.station_from_iata = segment.xpath(
            'ns1:departure_airport_code', namespaces=NSMAP)[0].text
        f.station_to_iata = segment.xpath(
            'ns1:arrival_airport_code', namespaces=NSMAP)[0].text

        departure = segment.xpath('ns1:departure_date', namespaces=NSMAP)[0].text + \
            segment.xpath('ns1:departure_time', namespaces=NSMAP)[0].text

        f.local_departure = datetime.strptime(departure, '%d.%m.%Y%H:%M')

        arrival = segment.xpath('ns1:arrival_date', namespaces=NSMAP)[0].text +\
            segment.xpath('ns1:arrival_time', namespaces=NSMAP)[0].text
        f.local_arrival = datetime.strptime(arrival, '%d.%m.%Y%H:%M')

        f.company_iata = segment.xpath('ns1:ak', namespaces=NSMAP)[0].text

        f.number = f.company_iata + ' ' + \
            segment.xpath('ns1:flight_number', namespaces=NSMAP)[0].text

        f.electronic_ticket = False

        flights.append(f)

    return flights
