# -*- coding: utf-8 -*-
import logging
import urllib
from datetime import datetime

import requests
from lxml import etree

from travel.avia.library.python.common.utils.safe_xml_parser import safe_xml_fromstring
from travel.avia.ticket_daemon.ticket_daemon.api.flights import Variant, FlightFabric, Flight
from travel.avia.ticket_daemon.ticket_daemon.daemon.utils import sleep_every, BadPartnerResponse, PartnerErrorTypes
from travel.avia.ticket_daemon.ticket_daemon.lib.baggage import Baggage
from travel.avia.ticket_daemon.ticket_daemon.lib.currency import Price
from travel.avia.ticket_daemon.ticket_daemon.lib.decorators import elementwise
from travel.avia.ticket_daemon.ticket_daemon.lib.http import url_complement_missing
from travel.avia.ticket_daemon.ticket_daemon.lib.tracker import QueryTracker
from travel.avia.ticket_daemon.ticket_daemon.lib.utils import skip_None_values
from travel.avia.ticket_daemon.ticket_daemon.lib.partner_secret_storage import partner_secret_storage

log = logging.getLogger(__name__)

CLIENT_ID = '4'
API_URL = 'https://meta.aviakassa.com/v4/meta/yandex'
API_KEY = partner_secret_storage.get(importer_name='aviakassa5', namespace='PASSWORD')
PARTNER_CODE = 'aviakass'

# подсмотретно внутри Marshmallow, универсальный способ находить булево значение в строке
truthy = frozenset(('t', 'T', 'true', 'True', 'TRUE', '1', 1, True))


def utms(national_version, is_charter):
    return {
        'utm_source': (
            'yandrasp_kz'
            if national_version == 'kz' else
            'yandexrasp'
        ),
        'utm_medium': 'cpc',
        'utm_campaign': 'charter' if is_charter else 'regular_flights'
    }


def validate_query(q):
    q.validate_passengers(adults=4, children=4)


@QueryTracker.init_query
def query(tracker, q):
    xml = get_data(tracker, q)
    variants = parse_response(xml, q)

    return variants


def get_data(tracker, q):
    adults = q.passengers.get('adults', 0)
    children = q.passengers.get('children', 0)
    infants = q.passengers.get('infants', 0)

    r = tracker.wrap_request(
        requests.post,
        API_URL,
        data=skip_None_values({
            'departure': q.iata_from,
            'arrival': q.iata_to,
            'date_forward': q.date_forward.strftime('%Y-%m-%d'),
            'date_backward': (q.date_backward.strftime('%Y-%m-%d')
                              if q.date_backward else None),
            'class': {'economy': 'E', 'business': 'B'}[q.klass],
            'adults': adults if adults else None,
            'children': children if children else None,
            'infants': infants if infants else None,
            'client_id': CLIENT_ID,
            'api_key': API_KEY,
        }),
        verify=False
    )

    content = r.text.encode('utf-8')

    if r.headers['content-type'] != 'application/xml':
        log.error('Bad content-type "%s": %s', r.headers['content-type'], content)
        raise BadPartnerResponse(PARTNER_CODE, r)

    xml = safe_xml_fromstring(content)

    check_errors(xml, r)

    return xml


def check_errors(xml_tree, response):
    errors = xml_tree.xpath('/Error')
    for error in errors:
        log.error('Partner server error: %s', response.content)
        if error.get('code') == '7':
            raise BadPartnerResponse(PARTNER_CODE, response, errors=PartnerErrorTypes.DATE_IN_THE_PAST)
        elif error.get('code') == '6':
            raise BadPartnerResponse(PARTNER_CODE, response, errors=PartnerErrorTypes.IVALID_REQUEST_STRUCTURE)
        raise BadPartnerResponse(PARTNER_CODE, response, errors=PartnerErrorTypes.ERROR)


def parse_response(xml, q):
    variants = []

    for v_tag in sleep_every(xml.xpath('variant')):

        v = Variant()

        fare_tag = v_tag.find('fare')
        if fare_tag is None:
            continue
        selfconnect = fare_tag.get('selfconnect') in truthy

        v.forward.segments = parse_flights(
            v_tag.xpath('route_forward'), q.importer.flight_fabric, selfconnect,
        )

        if q.date_backward:
            v.backward.segments = parse_flights(
                v_tag.xpath('route_backward'), q.importer.flight_fabric, selfconnect,
            )

        v.klass = {'E': 'economy', 'B': 'business'}[fare_tag.get('class')]

        currency = 'RUR'
        v.charter = fare_tag.get('charter') in truthy
        v.block = fare_tag.get('block') in truthy

        v.tariff = Price(float(fare_tag.get('value')), currency)

        v.url = url_complement_missing(
            urllib.unquote(v_tag.get('url')),
            params=utms(
                national_version=q.national_version,
                is_charter=v.charter
            )
        )

        v.order_data = {'url': v.url}

        variants.append(v)

    return variants


def get_baggage(fare_tag):
    try:
        pieces = fare_tag.get('pieces_of_luggage')
        if pieces in {None, ''}:
            return Baggage.from_partner()
        elif pieces == '0':
            return Baggage.from_partner(pieces=0)
        else:
            weight = fare_tag.get('luggage_weight')
            if weight:
                return Baggage.from_partner(pieces=int(pieces), weight=int(weight))
            return Baggage.from_partner(pieces=int(pieces))
    except Exception as exc:
        log.error(
            'Baggage parsing exception: {} on element {}'.format(exc, etree.tostring(fare_tag)))
        return Baggage.from_partner()


def parse_datetime(time):
    return datetime.strptime(time, '%Y-%m-%d %H:%M')


@elementwise
def parse_flights(f_tag, flight_fabric, selfconnect):
    # type: (etree.Element, FlightFabric, bool) -> Flight
    return flight_fabric.create(
        company_iata=f_tag.get('company_code'),
        pure_number=f_tag.get('route_code'),
        station_from_iata=f_tag.get('departure_airport_code'),
        station_to_iata=f_tag.get('arrival_airport_code'),
        local_departure=parse_datetime(f_tag.get('departure_datetime')),
        local_arrival=parse_datetime(f_tag.get('arrival_datetime')),
        baggage=get_baggage(f_tag),
        selfconnect=selfconnect,
        fare_code=f_tag.get('farecode'),
    )
