# -*- coding: utf-8 -*-
from datetime import datetime
from logging import getLogger

import requests
from lxml import etree

from travel.avia.ticket_daemon.ticket_daemon.lib.decorators import pipe
from travel.avia.ticket_daemon.ticket_daemon.lib.http import update_query_string
from travel.avia.ticket_daemon.ticket_daemon.api.flights import Variant
from travel.avia.ticket_daemon.ticket_daemon.daemon.utils import sleep_every
from travel.avia.ticket_daemon.ticket_daemon.lib.currency import Price
from travel.avia.ticket_daemon.ticket_daemon.lib.tracker import QueryTracker
from travel.avia.ticket_daemon.ticket_daemon.lib.baggage import BaggageParser
from travel.avia.ticket_daemon.ticket_daemon.lib.partner_secret_storage import partner_secret_storage

log = getLogger(__name__)
baggage_parser = BaggageParser(logger=log)

URL_TRACKER = {'from': 'yandextickets'}

AVIAOPERATOR_URL = 'http://api.aviaoperator.com/api/Search'
AVIAOPERATOR_PASSWORD = partner_secret_storage.get(
    importer_name='aviaoperator2', namespace='PASSWORD'
)


@QueryTracker.init_query
def query(tracker, q):
    # Партнер умеет отдавать только классы эконом и бизнес
    if q.klass not in ['economy', 'business']:
        return []

    xml = get_data(tracker, q)
    variants = parse_response(xml, q)

    return variants


def get_data(tracker, q):
    r = tracker.wrap_request(
        requests.get,
        AVIAOPERATOR_URL,
        params=build_search_query(q)
    )

    return r.content


def build_search_query(q):
    params = {}
    params['from'] = q.iata_from
    params['to'] = q.iata_to
    params['date1'] = q.date_forward.strftime('%Y-%m-%d')
    if q.date_backward:
        params['date2'] = q.date_backward.strftime('%Y-%m-%d')
    params['adults'] = q.passengers.get('adults', 0)
    params['children'] = q.passengers.get('children', 0)
    params['infants'] = q.passengers.get('infants', 0)
    params['cabin'] = {'economy': 'Y', 'business': 'C'}[q.klass]
    params['password'] = AVIAOPERATOR_PASSWORD

    return params


def parse_response(xml, q):
    log.debug(u'response %s', xml)
    parser = etree.XMLParser(recover=True)
    tree = etree.fromstring(xml, parser=parser)

    variants = []

    for e in sleep_every(tree.xpath('variant')):

        v = Variant()

        try:
            segments = e.xpath('segment')

            v.forward.segments = parse_flights(segments[0], q.importer.flight_fabric)
            has_backward = bool(q.date_backward and len(segments) > 1)
            if has_backward:
                v.backward.segments = parse_flights(segments[1], q.importer.flight_fabric)
            cabins = {ftag.find('cabin').text
                      for segment_tag in segments[:1 + has_backward]
                      for ftag in segment_tag.xpath('flight')}

            if len(cabins) != 1:
                continue

            v.klass = {'Y': 'economy', 'C': 'business'}[cabins.pop()]

            if e.find('currency').text.upper() != 'RUB':
                continue

            v.tariff = Price(float(e.find('price').text), 'RUR')

            v.url = update_query_string(e.find('url').text, URL_TRACKER)
            v.order_data = {'url': v.url}

        except Exception:
            log.warning('Error while parse variant')
            continue

        v.charter = True

        variants.append(v)

    return variants


@pipe(list)
def parse_flights(segment_tag, flight_fabric):
    for ftag in segment_tag.xpath('flight'):
        yield flight_fabric.create(
            station_from_iata=ftag.findtext('departure'),
            station_to_iata=ftag.findtext('arrival'),

            local_departure=datetime.strptime(
                ftag.findtext('departureDate') + ftag.findtext('departureTime'),
                '%Y-%m-%d%H:%M'
            ),
            local_arrival=datetime.strptime(
                ftag.findtext('arrivalDate') + ftag.findtext('arrivalTime'),
                '%Y-%m-%d%H:%M'
            ),
            company_iata=ftag.findtext('operatingCarrier'),
            number=ftag.findtext('number'),
            baggage=baggage_parser.parse_from_string(ftag.findtext('baggage'))
        )
