# -*- coding: utf-8 -*-
import re
from datetime import datetime
from logging import getLogger
from urllib import urlencode

import requests
from lxml import etree

from travel.avia.ticket_daemon.ticket_daemon.lib.decorators import pipe
from travel.avia.ticket_daemon.ticket_daemon.api.flights import Variant
from travel.avia.ticket_daemon.ticket_daemon.daemon.utils import sleep_every
from travel.avia.ticket_daemon.ticket_daemon.lib.currency import Price
from travel.avia.ticket_daemon.ticket_daemon.lib.tracker import QueryTracker
from travel.avia.ticket_daemon.ticket_daemon.lib.baggage import Baggage

log = getLogger(__name__)

BILETDV_URL = 'http://fares.biletdv.ru/SASSirenaFares.asmx/getFares'
D_FMT = '%d.%m.%YT%H:%M'
BAGGAGE_FORMAT = re.compile(r'(\d+)[,.]?\d*(\w+)', re.UNICODE)
KILOGRAMS = frozenset({u'кг', 'K', 'KG'})


@QueryTracker.init_query
def query(tracker, q):
    xml = get_data(tracker, q)
    variants = parse_response(xml, q)
    return variants


def get_data(tracker, q):
    params = {}

    # код города отправления (ИАТА), допускается ввод названия города.
    params['depCity'] = q.iata_from.encode('utf-8')

    # код города назначения (ИАТА), допускается ввод названия города.
    params['destCity'] = q.iata_to.encode('utf-8')

    # дата вылета в формате ДДММГГ или ДДММ
    params['depDate'] = q.date_forward.strftime('%d%m%y')

    # дата вылета обратно
    params['backDate'] = q.date_backward and \
        q.date_backward.strftime('%d%m%y') or ''

    # количество взрослых пассажиров (в количестве от 1 до 9).
    params['aaa'] = q.passengers.get('adults', 0)

    # количество детей от 2 до 12 лет (в количестве от 0 до 8).
    params['rbg'] = q.passengers.get('children', 0)

    # количество детей до 2 лет, на каждого взрослого не более одного
    params['rmg'] = q.passengers.get('infants', 0)

    params['direct'] = ''
    params['classes'] = {
        'economy': 'Э',
        'business': 'Б',
        'first': 'П',
    }.get(q.klass, 'Э')

    params['PartnerID'] = 'Yandex'

    r = tracker.wrap_request(
        requests.post,
        BILETDV_URL,
        headers={
            'Content-Type': 'application/x-www-form-urlencoded',
        },
        data=urlencode(params)
    )

    return r.content


@pipe(list)
def parse_response(xml, q):
    tree = etree.fromstring(xml.replace('xmlns="http://biletdv.ru/"', ''))

    is_success = tree.xpath("isSuccess")
    if not is_success or not is_success[0].text == 'true':
        return

    for p in sleep_every(tree.xpath('//Offers/Proposal')):
        v = Variant()

        v.tariff = Price(float(p.get('Total').replace(',', '.')))

        segments_out = p.xpath('Flights/Flight[@Direction=0]')

        if segments_out:
            v.forward.segments = parse_flights(segments_out, q.importer.flight_fabric)

        if not v.forward.segments:
            continue

        segments_in = p.xpath('Flights/Flight[@Direction="1"]')
        if segments_in:
            v.backward.segments = parse_flights(segments_in, q.importer.flight_fabric)

        v.url = p.find('BookURL').text

        v.klass = q.klass

        v.order_data = {
            'url': v.url
        }
        yield v


def get_baggage(segment):
    baggage_info = segment.get('BaggageRate')
    try:
        if baggage_info is None:
            return Baggage.from_partner()
        elif baggage_info == u'Нет':
            return Baggage.from_partner(included=False)
        else:
            count, desc = BAGGAGE_FORMAT.search(baggage_info).groups()
            if desc.startswith(u'мест'):
                return Baggage.from_partner(pieces=int(count))
            elif desc == u'PC':
                return Baggage.from_partner(pieces=int(count))
            elif desc in KILOGRAMS:
                return Baggage.from_partner(weight=int(count))
            else:
                log.error('Unknown baggage format %s', baggage_info)
    except Exception, exc:
        log.error('Baggage parsing exception: %s on element %s', exc, baggage_info)
    return Baggage.from_partner()


@pipe(list)
def parse_flights(segments, flight_fabric):
    for segment in segments:
        yield flight_fabric.create(
            station_from_iata=segment.get('Origin'),
            station_to_iata=segment.get('Destination'),
            local_departure=datetime.strptime(segment.get('Departure'), D_FMT),
            local_arrival=datetime.strptime(segment.get('Arrival'), D_FMT),
            company_iata=segment.get('Code'),
            fare_code=segment.get('FareCode'),
            pure_number=segment.get('Num'),
            baggage=get_baggage(segment),
        )
