# -*- coding: utf-8 -*-
from datetime import datetime
from logging import getLogger

import requests
from django.conf import settings
from lxml import etree

from travel.avia.ticket_daemon.ticket_daemon.api.flights import IATAFlight, Variant
from travel.avia.ticket_daemon.ticket_daemon.daemon.utils import sleep_every
from travel.avia.ticket_daemon.ticket_daemon.lib.currency import Price
from travel.avia.ticket_daemon.ticket_daemon.lib.tracker import QueryTracker

log = getLogger(__name__)

BILET_EXPERT_URL = 'http://www.biletexpert.ru/api/be_api.ashx?CP=yandex'


def validate_query(q):
    q.validate_country_codes()


@QueryTracker.init_query
def query(tracker, q):
    xml = get_data(tracker, q)

    variants = parse_response(xml, q)

    return variants


klass_map = {'Y': 1, 'C': 2, 'F': 3}
klass_to_letter = {'business': 'B', 'economy': 'E', 'first': 'F'}
klass_reverse_map = {1: 'economy', 2: 'business', 3: 'first'}


def parse_response(xml, q):
    tree = etree.fromstring(xml)

    variants = []

    for flight_tag in sleep_every(tree.findall('./Flight')):
        variant = parse_flight_tag(flight_tag, q.klass)

        if variant:
            variants.append(variant)

    return variants


def parse_flight_tag(flight_tag, klass):
    v = Variant()
    if settings.UNITTEST:
        v.raw_data = etree.tostring(
            flight_tag, pretty_print=True, encoding='utf8').decode('utf8')

    v.tariff = Price(get_tariff(flight_tag))

    v.forward.segments = parse_flights(flight_tag.find('./Out'))

    if flight_tag.find('./In') is not None:
        v.backward.segments = parse_flights(flight_tag.find('./In'))

    # В ответе нет указания класс, берем класс запроса
    v.klass = klass

    v.order_data = {
        'url': flight_tag.find('./Url').text
    }

    # если класс варианта не соответствует запросу, то пропускаем
    if v.klass == klass:
        return v


def parse_flights(inout_tag):
    flights = []
    for flight_info in inout_tag.findall('./FlightInfos'):
        f = IATAFlight()

        f.local_departure = datetime.strptime(
            flight_info.find('./StartDateTime').text, '%d.%m.%Y %H:%M')
        f.local_arrival = datetime.strptime(
            flight_info.find('./EndDateTime').text, '%d.%m.%Y %H:%M')

        f.station_from_iata = flight_info.find('./StartAirportCode').text
        f.station_to_iata = flight_info.find('./EndAirportCode').text

        f.company_iata = flight_info.find('./CarrierCode').text

        f.number = '%s %s' % (
            f.company_iata, flight_info.find('./FlightNumber').text)

        f.electronic_ticket = True

        flights.append(f)

    return flights


def get_tariff(flight_tag):
    amount = flight_tag.find('./FareInfos/CardTotalAmount')
    return float(amount.text)


def get_data(tracker, q):
    query_data = u"""<?xml version="1.0" encoding="UTF-8"?>
<FlightsSearch>
  <Origin>%(origin)s</Origin>
  <Destination>%(destination)s</Destination>
  <ServiceClass>%(class)s</ServiceClass>
  <FlightType>%(flight_type)s</FlightType>
  <StartDate>%(start_date)s</StartDate>
  <ReturnDate>%(return_date)s</ReturnDate>
  <ADTCnt>%(ADTQnt)d</ADTCnt>
  <CNNCnt>%(CHDQnt)d</CNNCnt>
  <INFCnt>%(INFQnt)d</INFCnt>
  <DFO>N</DFO>
</FlightsSearch>""" % {
        'origin': q.iata_from,
        'destination': q.iata_to,
        'class': klass_to_letter[q.klass],
        'flight_type': 'RT' if q.date_backward else 'OW',
        'start_date': q.date_forward.strftime('%d.%m.%Y'),
        'return_date': q.date_backward.strftime('%d.%m.%Y') if q.date_backward else '',
        'ADTQnt': q.passengers.get('adults', 0),
        'CHDQnt': q.passengers.get('children', 0),
        'INFQnt': q.passengers.get('infants', 0),
    }

    r = tracker.wrap_request(
        requests.post,
        BILET_EXPERT_URL,
        headers={
            'Accept-Encoding': 'gzip',
        },
        data=query_data.encode('utf8')
    )

    return r.content
