# -*- coding: utf-8 -*-
import ujson
from datetime import datetime

import requests
from typing import Dict

from travel.avia.ticket_daemon.ticket_daemon.api.query import Query
from travel.avia.ticket_daemon.ticket_daemon.api.flights import Variant
from travel.avia.ticket_daemon.ticket_daemon.daemon.utils import sleep_every
from travel.avia.ticket_daemon.ticket_daemon.lib.currency import Price
from travel.avia.ticket_daemon.ticket_daemon.lib.tracker import QueryTracker
from travel.avia.ticket_daemon.ticket_daemon.lib.partner_secret_storage import partner_secret_storage

BILETIK_API_URL = 'http://search.biletik.aero/search'
BILETIK_TOKEN = partner_secret_storage.get(importer_name='biletik6', namespace='password')


@QueryTracker.init_query
def query(tracker, q):
    json_content, headers = get_data(tracker, q)
    variants = generate_variants(json_content, headers, q)
    return variants


def get_data(tracker, q):
    r = tracker.wrap_request(
        requests.post,
        BILETIK_API_URL,
        headers={
            'accept': 'application/json',
            'Authorization': 'Bearer ' + BILETIK_TOKEN,
            'Content-Type': 'application/json',
        },
        data=build_search_body(q)
    )

    return r.json(), r.headers


def prepare_segment(raw_data, flight_fabric):
    for segment in sleep_every(raw_data):
        station_from = segment['dep']['airport']
        station_to = segment['arr']['airport']
        departure = segment['dep']['dateTime']
        arrival = segment['arr']['dateTime']
        company = segment['operatingAirline']
        flight_number = segment['flightNumber']
        stop_points = segment.get('stopPoints')
        if stop_points:
            stops = filter(lambda i: i.get('depDateTime') and i.get('arrDateTime'), stop_points)
            for stop in stops:
                stop_station = stop['airport']
                stop_departure = stop['depDateTime']
                stop_arrival = stop['arrDateTime']
                yield flight_fabric.create(
                    station_from_iata=station_from,
                    station_to_iata=stop_station,
                    local_departure=parse_datetime(departure),
                    local_arrival=parse_datetime(stop_arrival),
                    company_iata=company,
                    pure_number=flight_number,
                )
                station_from = stop_station
                departure = stop_departure

        yield flight_fabric.create(
            station_from_iata=station_from,
            station_to_iata=station_to,
            local_departure=parse_datetime(departure),
            local_arrival=parse_datetime(arrival),
            company_iata=company,
            pure_number=flight_number,
        )


def generate_variants(results, headers, q):
    variants = []

    for legs in results:
        raw_data = legs['legs']
        v = Variant()
        v.forward.segments = list(prepare_segment(raw_data[0], q.importer.flight_fabric))
        if len(raw_data) > 1:
            v.backward.segments = list(prepare_segment(raw_data[1], q.importer.flight_fabric))

        v.klass = legs['cabinClass'].lower()
        v.url = headers['X-Redirect-URL'] + legs['token']
        v.order_data = {'url': v.url}
        v.tariff = Price(value=legs['cost'], currency=headers['X-Currency-Code'])

        variants.append(v)

    return variants


def build_search_body(q):
    # type: (Query) -> Dict
    request_parameters = {
        'routes': [
            {'depDate': q.date_forward.strftime('%Y-%m-%d'),
             'depPoint': q.point_from.iata,
             'arrPoint': q.point_to.iata,
             }
        ],
        'passengers': build_passengers(q),
        'cabinClass': q.klass.capitalize(),
    }

    if q.date_backward:
        request_parameters['routes'].append(
            {'depDate': q.date_backward.strftime('%Y-%m-%d'),
             'depPoint': q.point_to.iata,
             'arrPoint': q.point_from.iata,
             })

    return ujson.dumps(request_parameters)


def build_passengers(q):
    # type: (Query) -> Dict
    passengers = {'ADT': q.passengers['adults']}
    if q.passengers.get('children') > 0:
        passengers['CNN'] = q.passengers['children']
    if q.passengers.get('infants') > 0:
        passengers['INF'] = q.passengers['infants']
    return passengers


def parse_datetime(raw_datetime):
    return datetime.strptime(
        raw_datetime,
        '%Y-%m-%dT%H:%M:%S'
    )
