# -*- coding: utf-8 -*-

import hashlib
from datetime import datetime, timedelta
from itertools import product
from logging import getLogger

import requests
from django.template import loader
from lxml import etree

from travel.avia.ticket_daemon.ticket_daemon.api.flights import Variant
from travel.avia.ticket_daemon.ticket_daemon.daemon.utils import sleep_every
from travel.avia.ticket_daemon.ticket_daemon.lib.currency import Price
from travel.avia.ticket_daemon.ticket_daemon.lib.http import url_complement_missing
from travel.avia.ticket_daemon.ticket_daemon.lib.tracker import QueryTracker
from travel.avia.ticket_daemon.ticket_daemon.lib.baggage import Baggage
from travel.avia.ticket_daemon.ticket_daemon.lib.partner_secret_storage import partner_secret_storage

log = getLogger(__name__)

BILETIX_URL = 'http://search-api.biletix.ru/bitrix/components/travelshop/ibe.soap/travelshop_booking.php'
BILETIX_LOGIN = '[partner]||biletix.kz.yandex'
BILETIX_PASSWD = partner_secret_storage.get(
    importer_name='biletix_kz', namespace='PASSWORD'
)
BILETIX_SHARED_SECRET = partner_secret_storage.get(
    importer_name='biletix_kz', namespace='SHAREDSECRET'
)

KLASS_MAP = {
    'economy': u'E',
    'business': u'B',
    'first': u'F',
}

URL_TRACKER = {
    'utm_source': 'yandex.ru',
    'utm_medium': 'referral',
    'utm_campaign': 'metasearch_kz',
}


@QueryTracker.init_query
def query(tracker, q):
    xml = get_data(tracker, q)

    variants = list(parse_response(xml, q))

    return variants


def get_data(tracker, q):
    params = [
        ('session_token', BILETIX_LOGIN + ':' + BILETIX_PASSWD + '::KZT'),
        ('owrt', 'RT' if q.date_backward else 'OW'),
        ('departure_point', q.iata_from),
        ('arrival_point', q.iata_to),
        ('outbound_date', q.date_forward.strftime('%d.%m.%Y')), ]

    if q.date_backward:
        params += [('return_date', q.date_backward.strftime('%d.%m.%Y')), ]

    params += [
        ('adult_count', q.passengers.get('adults', 0)),
        ('child_count', q.passengers.get('children', 0)),
        ('infant_count', q.passengers.get('infants', 0)),
        ('class', KLASS_MAP.get(q.klass, 'E')),
    ]

    conc = u''.join([unicode(v) for n, v in params]) + BILETIX_SHARED_SECRET

    params.append(('hash', hashlib.md5(conc.encode('utf-8')).hexdigest()))

    query_xml = loader.render_to_string(
        'partners/biletix.xml', dict(params)
    ).strip()

    r = tracker.wrap_request(
        requests.post,
        BILETIX_URL,
        headers={
            'Content-Type': 'text/xml; charset=utf-8',
            'SOAPAction': 'urn:#GetOptimalFares',
        },
        data=query_xml.encode('utf-8')
    )

    return r.content


NSMAP = {'ns1': 'http://www.tais.ru/'}


def create_segments(flight_tag, direction, q):
    direction_xpath = 'ns1:directions/ns1:GetOptimalFaresDirection[ns1:direction="{}"]'.format(direction)
    direction = flight_tag.xpath(direction_xpath, namespaces=NSMAP)

    flights_to = direction and direction[0].xpath(
        'ns1:flights/ns1:GetOptimalFaresFlight', namespaces=NSMAP
    ) or []
    # Собрать сегменты для каждого полёта
    return [{
        'xml': f,
        'seg': list(parse_flights(
            q.importer.flight_fabric,
            f.xpath('ns1:segments/ns1:AirSegment', namespaces=NSMAP),
        ))
    } for f in flights_to]


def parse_response(xml, q):
    tree = etree.fromstring(xml)
    offers_tree = tree.xpath('//ns1:offers[1]', namespaces=NSMAP)[0]
    for flight_tag in sleep_every(offers_tree.xpath('ns1:GetOptimalFaresOffer', namespaces=NSMAP)):
        flights_to = create_segments(flight_tag, "TO", q)
        if not flights_to:
            continue
        flights_back = create_segments(flight_tag, "BACK", q) or [None]

        charter = flight_tag.findtext('ns1:charter', namespaces=NSMAP) == "Y"

        tariff = float(
            flight_tag.findtext('ns1:total_price', namespaces=NSMAP)
        )

        baseurl = flight_tag.findtext('ns1:link', namespaces=NSMAP)

        for flight_to, flight_back in product(flights_to, flights_back):
            v = Variant()
            v.forward.segments = flight_to['seg']
            if flight_back:
                v.backward.segments = flight_back['seg']

                # RASPTICKETS-388
                if v.backward.segments and \
                        v.forward.segments[-1].local_arrival and v.backward.segments[0].local_departure and \
                        v.forward.segments[-1].local_arrival + timedelta(hours=1) > v.backward.segments[0].local_departure:
                    continue

            v.tariff = Price(tariff, currency='KZT')

            v.url = baseurl + flight_to['xml'].findtext(
                'ns1:link', namespaces=NSMAP
            )
            if flight_back:
                v.url += flight_back['xml'].findtext(
                    'ns1:link', namespaces=NSMAP
                )

            v.url = url_complement_missing(v.url, URL_TRACKER)
            v.charter = charter
            v.klass = q.klass
            v.order_data = {'url': v.url}

            yield v


def get_baggage(segment_tag):
    try:
        bag_type = segment_tag.findtext('ns1:bagtype', namespaces=NSMAP)
        if bag_type is None:
            raise Exception('ns1:bagtype tag not found')

        if bag_type == 'NO':
            return Baggage.from_partner(included=False)
        else:
            bag_allowance_str = segment_tag.findtext('ns1:bagallowance', namespaces=NSMAP)
            if bag_allowance_str is None:
                raise Exception('ns1:bagallowance tag not found')

            bag_allowance = int(bag_allowance_str)
            if bag_type == 'PC':
                return Baggage.from_partner(pieces=bag_allowance)
            elif bag_type == 'KG':
                return Baggage.from_partner(weight=bag_allowance)
            else:
                log.info('Unknown baggage type %s', bag_type)
    except Exception, exc:
        log.info('Baggage parsing exception: %r on element %r', exc, segment_tag)
    return Baggage.from_partner()


def parse_datetime(raw_date, raw_time):
    """
    Example '21.01.2018' and '13:50'
    """
    return datetime(
        int(raw_date[6:10]), int(raw_date[3:5]), int(raw_date[0:2]),
        int(raw_time[0:2]), int(raw_time[3:5])
    )


def parse_flights(flight_fabric, segments):
    for segment in segments:
        fare_code = (segment.findtext('ns1:fare_code', namespaces=NSMAP)
                       or segment.findtext('ns1:fare_codes', namespaces=NSMAP))
        # Ignore all tariffs except the first one, see RASPTICKETS-21678 for details
        if fare_code:
            fare_code = fare_code.split(',')[0]

        yield flight_fabric.create(
            station_from_iata=segment.findtext(
                'ns1:departure_airport_code', namespaces=NSMAP),
            station_to_iata=segment.findtext(
                'ns1:arrival_airport_code', namespaces=NSMAP),
            local_departure=parse_datetime(
                segment.findtext('ns1:departure_date', namespaces=NSMAP),
                segment.findtext('ns1:departure_time', namespaces=NSMAP),
            ),
            local_arrival=parse_datetime(
                segment.findtext('ns1:arrival_date', namespaces=NSMAP),
                segment.findtext('ns1:arrival_time', namespaces=NSMAP),
            ),
            company_iata=segment.findtext('ns1:ak', namespaces=NSMAP),
            pure_number=segment.findtext('ns1:flight_number', namespaces=NSMAP),
            baggage=get_baggage(segment),
            fare_code=fare_code,
        )
