# -*- coding: utf-8 -*-
from datetime import datetime
from logging import getLogger
from urllib import urlencode
from uuid import uuid4

import requests
from collections import OrderedDict as SortedDict
from django.conf import settings
from lxml import etree

from travel.avia.ticket_daemon.ticket_daemon.api.flights import IATAFlight, Variant
from travel.avia.ticket_daemon.ticket_daemon.daemon.utils import sleep_every
from travel.avia.ticket_daemon.ticket_daemon.lib.currency import Price
from travel.avia.ticket_daemon.ticket_daemon.lib.tracker import QueryTracker
from travel.avia.ticket_daemon.ticket_daemon.lib.partner_secret_storage import partner_secret_storage

log = getLogger(__name__)

NSMAP = {'n': 'http://airts.ru/webservices/2009/01'}

BILET_ONLINE_CLIENT_ID = 'YAND'
BILETONLINE_PARTNER_ID = 310681945
BILETONLINE_QUERY_URL = 'http://www.bilet-on-line.ru/bolservices/soaBus'
BILETONLINE_ORDER_URL = 'http://www.bilet-on-line.ru/xworkview.htm'
BILETONLINE_PASSWORD = partner_secret_storage.get(
    importer_name='biletonline', namespace='PASSWORD'
)


def validate_query(q):
    q.validate_country_codes()


@QueryTracker.init_query
def query(tracker, q):
    xml = get_data(tracker, q)

    variants = parse_response(xml, q)

    return list(variants)


klass_map = {'Y': 1, 'C': 2, 'F': 3, 'S': 1}
klass_reverse_map = {1: 'economy', 2: 'business', 3: 'first'}
klass_code_to_letter = {'economy': 'Y', 'business': 'C', 'first': 'F'}


def parse_response(xml, q):
    log.debug(u"Разбор ответа")

    tree = etree.fromstring(xml)

    for priced_itinerary in sleep_every(tree.xpath('//n:PricedItinerary', namespaces=NSMAP)):
        variant = parse_variant(q, priced_itinerary)

        if variant:
            yield variant


def parse_variant(q, priced_itinerary):
    v = Variant()
    if settings.UNITTEST:
        v.raw_data = etree.tostring(
            priced_itinerary, encoding='utf8', pretty_print=True).decode('utf8')

    v.tariff = Price(float(get_tariff(priced_itinerary)))

    options = list(priced_itinerary.xpath('./n:AirItinerary/n:OriginDestinationOptions/n:OriginDestinationOption',
                                          namespaces=NSMAP))

    v.forward.segments = parse_flights(options[0])

    if not v.forward.segments:
        return None

    klass = None
    cabin = priced_itinerary.xpath('./n:AirItineraryPricingInfo/n:FareInfos/n:FareInfo/n:TPA_Extensions/n:Cabin',
                                   namespaces=NSMAP)
    if cabin:
        klass = klass_reverse_map[klass_map[cabin[0].get('Cabin')]]

    try:
        v.backward.segments = parse_flights(options[1]) or None
    except IndexError:
        pass

    if q.date_backward and not v.backward.segments:
        return None

    v.klass = klass or q.klass

    v.order_data = {
        'url': make_url(priced_itinerary),
    }

    # если класс варианта не соответствует запросу, то пропускаем
    if v.klass == q.klass:
        return v

    return None


def parse_flights(option):
    flights = []

    for flight_segment in option.xpath('./n:FlightSegment', namespaces=NSMAP):
        f = IATAFlight()

        f.local_departure = datetime.strptime(
            flight_segment.get('DepartureDateTime'), '%Y-%m-%dT%H:%M:%S')
        f.local_arrival = datetime.strptime(
            flight_segment.get('ArrivalDateTime'), '%Y-%m-%dT%H:%M:%S')

        f.station_from_iata = flight_segment.xpath('./n:DepartureAirport',
                                                   namespaces=NSMAP)[0].get('LocationCode')
        f.station_to_iata = flight_segment.xpath('./n:ArrivalAirport',
                                                 namespaces=NSMAP)[0].get('LocationCode')

        f.company_iata = flight_segment.xpath('./n:MarketingAirline',
                                              namespaces=NSMAP)[0].get('Code')

        f.number = '%s %s' % (
            f.company_iata, flight_segment.get('FlightNumber'))

        eticket = flight_segment.xpath('./n:TPA_Extensions/n:eTicket',
                                       namespaces=NSMAP)

        if eticket:
            f.electronic_ticket = eticket[0].get('Ind') == u'true'
        else:
            f.electronic_ticket = True

        flights.append(f)

    return flights


def get_tariff(priced_itinerary):
    total_fares = priced_itinerary.xpath(
        './/n:ItinTotalFare/n:TotalFare', namespaces=NSMAP)
    total_fare = total_fares[0]
    return total_fare.get('Amount')


def make_url(priced_itinerary):
    params = SortedDict((
        ('algorithm', 'AIR.SEARCH'),
        ('op', 'partner_goToPassangers'),
        ('partnerId', str(BILETONLINE_PARTNER_ID)),
        ('sequenceId', priced_itinerary.get('SequenceNumber', '')),
        ('posid', BILET_ONLINE_CLIENT_ID)
    ))

    return BILETONLINE_ORDER_URL + '?' + urlencode(params)


def get_data(tracker, q):
    header = make_header()
    query_data = header + u"""
    <SOAP-ENV:Body>
        <ATS_AirLowFareSearchRQ xmlns="http://airts.ru/webservices/2009/01" Version="2009.1">
            <POS>
                <Source TerminalID="%(client_id)s">
                    <BookingChannel>
                        <CompanyName Code="%(client_id)s" CodeContext="ATS"/>
                    </BookingChannel>
                </Source>
            </POS>
            %(destanation_info)s
            <TravelPreferences MaxStopsQuantity="1">
                <CabinPref PreferLevel="Preferred" Cabin="%(cabin)s"/>
                <TPA_Extensions>
                    <TripType Value="%(trip_type)s"/>
                </TPA_Extensions>
            </TravelPreferences>
            <TravelerInfoSummary>
                %(seats)s
                <PriceRequestInformation>
                    <TPA_Extensions>
                        <PointOfSaleOverride Code="MOW"/>
                    </TPA_Extensions>
                </PriceRequestInformation>
            </TravelerInfoSummary>
            <TPA_Extensions>
                <IntelliSellTransaction>
                    <RequestType Name="50ITINS"/>
                </IntelliSellTransaction>
            </TPA_Extensions>
        </ATS_AirLowFareSearchRQ>
    </SOAP-ENV:Body>
</SOAP-ENV:Envelope>""" % {
        'client_id': BILET_ONLINE_CLIENT_ID,
        'destanation_info': make_destanation_info(q),
        'trip_type': 'Return' if q.date_backward else 'OneWay',
        'seats': make_seats(q),
        'cabin': klass_code_to_letter[q.klass]
    }

    r = tracker.wrap_request(
        requests.post,
        BILETONLINE_QUERY_URL,
        data=query_data.encode('utf8')
    )

    return r.content


def make_destanation_info(q):
    result = u"""<OriginDestinationInformation RPH="1">
    <DepartureDateTime>%(departure)s</DepartureDateTime>
    <OriginLocation LocationCode="%(iata_from)s" CodeContext="IATA"/>
    <DestinationLocation LocationCode="%(iata_to)s" CodeContext="IATA"/>
</OriginDestinationInformation>
""" % {
        'departure': q.date_forward.strftime('%Y-%m-%dT00:00:00'),
        'iata_from': q.iata_from,
        'iata_to': q.iata_to
    }
    if q.date_backward:
        result += u"""<OriginDestinationInformation RPH="2">
    <DepartureDateTime>%(departure)s</DepartureDateTime>
    <OriginLocation LocationCode="%(iata_from)s" CodeContext="IATA"/>
    <DestinationLocation LocationCode="%(iata_to)s" CodeContext="IATA"/>
</OriginDestinationInformation>
""" % {
            'departure': q.date_backward.strftime('%Y-%m-%dT00:00:00'),
            'iata_from': q.iata_to,
            'iata_to': q.iata_from
        }

    return result


def make_header():
    today = datetime.today()
    message_id = today.strftime('%Y%m%d%H%M%S') + '-' + str(uuid4().int)
    return u"""<?xml version="1.0" encoding="utf-8"?>
    <SOAP-ENV:Envelope xmlns:SOAP-ENV="http://schemas.xmlsoap.org/soap/envelope/" xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance" xmlns:xsd="http://www.w3.org/2001/XMLSchema">
    <SOAP-ENV:Header>
        <m:MessageHeader xmlns:m="http://www.ebxml.org/namespaces/messageHeader" version="">
            <m:From>
                <m:PartyId>%(client_id)s</m:PartyId>
                <m:Role>SRV</m:Role>
            </m:From>
            <m:To>
                <m:PartyId>ATS</m:PartyId>
                <m:Role>MOWG</m:Role>
            </m:To>
            <m:CPAId>%(client_id)s</m:CPAId>
            <m:CPAKey>%(password)s</m:CPAKey>
            <m:Service type="ATS">MobileReservation</m:Service>
            <m:Action>ATS_AirLowFareSearchRQ</m:Action>
            <m:MessageData>
                <m:MessageId>%(message_id)s</m:MessageId>
                <m:Timestamp>%(timestamp)s</m:Timestamp>
            </m:MessageData>
        </m:MessageHeader>
    </SOAP-ENV:Header>""" % {
        'client_id': BILET_ONLINE_CLIENT_ID,
        'password': BILETONLINE_PASSWORD,
        'timestamp': today.strftime('%Y-%m-%dT%H:%M:%S'),
        'message_id': message_id

    }


def make_seats(q):
    total = 0
    passangers_template = u'<PassengerTypeQuantity Code="%(code)s" Quantity="%(quantity)s"/>\n'
    passangers = ""
    if q.passengers.get('adults', 0) != 0:
        total += q.passengers.get('adults', 0)
        passangers += passangers_template % {
            'code': 'ADT',
            'quantity': q.passengers.get('adults', 0),
        }

# =========================================================================
# >Детей в возрасте от 2-х до 12 лет следует кодировать как Сnn , где вместо NN – нужно указывать
# >число полных лет на дату выполнения обратного рейса (напр. C06, С07, С08, С09, С10, С11 и С12)
# мы будем указывать 9 летних
# =========================================================================

    if q.passengers.get('children', 0) != 0:
        total += q.passengers.get('children', 0)
        passangers += passangers_template % {
            'code': 'C09',
            'quantity': q.passengers.get('children', 0),
        }

# =========================================================================
# В элементе SeatsRequested, клиенту следует указывать число требуемых мест (помните, что
# число мест не включает в себя младенцев без мест (INF)).
# =========================================================================

    if q.passengers.get('infants', 0) != 0:
        passangers += passangers_template % {
            'code': 'INF',
            'quantity': q.passengers.get('infants', 0),
        }

    passangers_template
    return u"""<SeatsRequested>%(total)s</SeatsRequested>
<AirTravelerAvail>
    %(passangers)s
</AirTravelerAvail>
""" % {
        'total': total,
        'passangers': passangers
    }
