# -*- coding: utf-8 -*-
import logging
import re
from datetime import datetime
from urllib import urlencode

import requests
from lxml import etree

from travel.avia.ticket_daemon.ticket_daemon.api.flights import Variant, Baggage
from travel.avia.ticket_daemon.ticket_daemon.daemon.utils import BadPartnerResponse, sleep_every
from travel.avia.ticket_daemon.ticket_daemon.lib.currency import Price
from travel.avia.ticket_daemon.ticket_daemon.lib.decorators import elementwise, on_exception
from travel.avia.ticket_daemon.ticket_daemon.lib.tracker import QueryTracker
from travel.avia.ticket_daemon.ticket_daemon.lib.partner_secret_storage import partner_secret_storage

log = logging.getLogger(__name__)

PARTNER_CODE = 'bookandtrip'

SEARCH_URL = 'http://api.bookandtrip.ru/avia.php'
API_KEYS_BY_NATIONAL_VERSION = {
    'ua': partner_secret_storage.get(
        importer_name='bookandtrip', namespace='UAPASSWORD'
    ),
    'ru': partner_secret_storage.get(
        importer_name='bookandtrip', namespace='RUPASSWORD'
    ),
}
DRIVER_BY_NATIONAL_VERSION = {
    'ua': 'xml_ua',
    'ru': 'xml_ru',
}
REF_ID_BY_NATIONAL_VERSION = {
    'ua': 468,
    'ru': 466,
}

KLASS_MAP = {'economy': 'E', 'business': 'B'}
ERROR_MSG = 'We are unable to find recommendations for your search. Please change your search criteria and resubmit the search.'


def validate_query(q):
    q.validate_passengers(4, 4, 1)
    q.validate_klass(KLASS_MAP)


@QueryTracker.init_query
def query(tracker, q):
    r = get_response(tracker, q)
    variants = parse_response(r, q)

    return variants


def format_date(d):
    return d.strftime('%d-%m-%Y')


def get_response(tracker, q):
    params = {
        'action': 'search',
        'driver': DRIVER_BY_NATIONAL_VERSION.get(q.national_version),
        'key': API_KEYS_BY_NATIONAL_VERSION.get(q.national_version),
        'adt': q.passengers.get('adults', 0),
        'chd': q.passengers.get('children', 0),
        'inf': q.passengers.get('infants', 0),
        'service_class': KLASS_MAP[q.klass],

        'destinations[0][departure]': q.iata_from,
        'destinations[0][arrival]': q.iata_to,
        'destinations[0][date]': format_date(q.date_forward),
    }

    if q.date_backward:
        params.update({
            'destinations[1][departure]': q.iata_to,
            'destinations[1][arrival]': q.iata_from,
            'destinations[1][date]': format_date(q.date_backward),
        })

    r = tracker.wrap_request(
        requests.get,
        SEARCH_URL,
        params=params
    )

    return r


def parse_response(r, q):
    tree = etree.fromstring(r.text.encode('utf-8'))
    variants = []

    error = tree.xpath('ERRORS/ERROR')
    if error:
        if error[0].text == ERROR_MSG:
            return []
        else:
            raise BadPartnerResponse(PARTNER_CODE, r, errors=error[0].text)

    currency = tree.findtext('CURRENCY')

    redirect_url = tree.findtext('REDIRECT_URL') + '&' + urlencode({
        'class': KLASS_MAP[q.klass],
        'refid': REF_ID_BY_NATIONAL_VERSION.get(q.national_version),
        'act': 'book',
    })

    for v_tag in sleep_every(tree.xpath('ITEMS/ITEM')):
        v = Variant()

        routes = v_tag.xpath('TRIP_PART')

        try:
            v.forward.segments = parse_flight(routes[0].xpath('SEGMENT'), q)

            if q.date_backward and len(routes) > 1:
                v.backward.segments = parse_flight(
                    routes[1].xpath('SEGMENT'), q)

        except Exception as e:
            log.error('Parsing error: %s', e)
            continue

        v.klass = q.klass

        price_val = float(v_tag.get('amount'))
        v.tariff = Price(price_val, currency)

        r_hash = v_tag.get('hash')

        v.url = redirect_url + '&' + urlencode({
            'r_hash': r_hash.encode('utf-8'),
        })

        v.order_data = {'url': v.url}

        variants.append(v)

    return variants


baggage_regexp = re.compile(r'(\d+)(\w+)')
baggage_seat_types = {'n', 'p', 'pc'}
baggage_weight_type = {'k', 'kg'}


def _get_baggage(raw_baggage):
    try:
        if not raw_baggage:
            return Baggage.from_partner()

        raw_baggage = raw_baggage.lower()
        if raw_baggage == 'n/a':
            return Baggage.from_partner()
        if raw_baggage in {'nil', 'no'}:
            return Baggage.from_partner(included=False)

        # 1N/1N - до слэша, касается взрослых пассажиров, после инфант.
        # !!! учитываем только багаж для взрослых, так как пока не умеем различать багажи для детей и взрослых
        raw_baggage = raw_baggage.split('/')[0]

        groups = baggage_regexp.findall(raw_baggage)
        assert len(groups) == 1

        baggage_value, baggage_type = groups[0]
        baggage_value = int(baggage_value)

        if baggage_type in baggage_seat_types:
            return Baggage.from_partner(pieces=baggage_value)
        if baggage_type in baggage_weight_type:
            return Baggage.from_partner(weight=baggage_value)
    except Exception:
        log.exception('Baggage parsing error: %s', raw_baggage)
    return Baggage.from_partner()


def parse_datetime(raw_date):
    """
    Example '21.01.2018 13:50' for raw_date
    """
    return datetime(
        int(raw_date[6:10]), int(raw_date[3:5]), int(raw_date[0:2]),
        int(raw_date[11:13]), int(raw_date[14:16])
    )


FLIGHT_KLASS_MAP = {
    'economy': 'economy',
    'business': 'business',
    'premiumeconomy': 'economy',
}


@elementwise
@on_exception(lambda e, fn, f_tag, q: log.error(
    '%r in %s %s with f_tag: %s',
    e, fn.__name__, q.key(), etree.tostring(f_tag))
)
def parse_flight(f_tag, q):
    dep = f_tag.xpath('DEPARTURE')[0]
    arr = f_tag.xpath('ARRIVAL')[0]
    flight_tag = f_tag.xpath('FLIGHT')[0]

    return q.importer.flight_fabric.create(
        station_from_iata=dep.get('airport') or dep.get('city'),
        station_to_iata=arr.get('airport') or arr.get('city'),
        local_departure=parse_datetime(dep.text),
        local_arrival=parse_datetime(arr.text),
        company_iata=flight_tag.get('supplier_code'),
        pure_number=flight_tag.text,
        klass=FLIGHT_KLASS_MAP[flight_tag.get('service_class')],
        baggage=_get_baggage(flight_tag.get('baggage'))
    )
