# -*- coding: utf-8 -*-
import urllib
from datetime import datetime
from logging import getLogger
from urllib import urlencode

import requests
from django.template import loader
from lxml import etree

from travel.avia.ticket_daemon.ticket_daemon.api.flights import IATAFlight, Variant
from travel.avia.ticket_daemon.ticket_daemon.daemon.utils import sleep_every
from travel.avia.ticket_daemon.ticket_daemon.lib.currency import Price
from travel.avia.ticket_daemon.ticket_daemon.lib.tracker import QueryTracker

log = getLogger(__name__)

KLASS_MAP = {'economy': u'Y', 'business': u'C', 'first': u'F'}
KLASS_REVERSE_MAP = {value: key for key, value in KLASS_MAP.items()}
SERVICE_CLASS_PRIORITIES = {'economy': 1, 'business': 2, 'first': 3}

SEARCH_URL = 'http://api.booker.ru/air/Search'
PARTNER_ID = 'yandex'


def validate_query(q):
    q.validate_klass(KLASS_MAP)


@QueryTracker.init_query
def query(tracker, q):
    xml = get_data(tracker, q)

    variants = parse_response(xml, q)

    return variants


def build_aviasearch_params(q):
    params = {
        'from': q.iata_from.encode('utf-8'),
        'to': q.iata_to.encode('utf-8'),
        'date1': q.date_forward.strftime('%Y-%m-%d'),
        'adults': q.passengers.get('adults', 0),
        'children': q.passengers.get('children', 0),
        'infants': q.passengers.get('infants', 0),
        'cabin': KLASS_MAP[q.klass],
        'partner': PARTNER_ID,
    }

    if q.date_backward:
        params['date2'] = q.date_backward.strftime('%Y-%m-%d')

    return params


def build_xml(xml_template_file, params):
    query_xml = loader.render_to_string(xml_template_file, params)

    return ''.join(filter(None, [
        line.strip() for line in query_xml.splitlines()
    ]))


def get_data(tracker, q):
    url = SEARCH_URL + '?' + urlencode(build_aviasearch_params(q))

    r = tracker.wrap_request(
        requests.get,
        url,
    )

    return r.content


def parse_response(xml, q):
    tree = etree.fromstring(xml)
    variants = []

    for variant in sleep_every(tree.xpath('//variants/variant')):
        v = Variant()

        price = variant.find('price').text
        variant_currency = variant.find('currency').text

        currency = {'RUB': 'RUR'}.get(variant_currency, variant_currency)

        v.tariff = Price(float(price), currency=currency)

        v.url = urllib.unquote(variant.find('url').text)

        segments = variant.xpath('segment')

        v.forward.segments = parse_segments(segments[0])

        if q.date_backward:
            v.backward.segments = parse_segments(segments[1])

        v.order_data = {'url': v.url}

        sorted_classes = sorted(v.klasses, key=SERVICE_CLASS_PRIORITIES.get)

        v.klass = sorted_classes[0] if sorted_classes else q.klass

        variants.append(v)

    return variants


def parse_segments(xpath_segments):
    segments = []

    flights = xpath_segments.xpath('flight')

    for flight in flights:
        f = IATAFlight()

        f.station_from_iata = flight.find('departure').text
        f.station_to_iata = flight.find('arrival').text

        f.local_departure = datetime.strptime(
            '%s %s' % (
                flight.find('departureDate').text,
                flight.find('departureTime').text,
            ),
            '%Y-%m-%d %H:%M'
        )

        f.local_arrival = datetime.strptime(
            '%s %s' % (
                flight.find('arrivalDate').text,
                flight.find('arrivalTime').text,
            ),
            '%Y-%m-%d %H:%M'
        )

        f.company_iata = flight.find('operatingCarrier').text

        if f.company_iata:
            f.number = '%s %s' % (
                f.company_iata,
                flight.find('number').text
            )

        else:
            f.number = flight.find('number').text

        f.electronic_ticket = False

        f.klass = KLASS_REVERSE_MAP.get(flight.find('cabin').text, None)

        segments.append(f)

    return segments
