# -*- coding: utf-8 -*-
from datetime import datetime
from logging import getLogger

import requests
from django.template import loader
from lxml import etree

from travel.avia.ticket_daemon.ticket_daemon.api.flights import IATAFlight, Variant
from travel.avia.ticket_daemon.ticket_daemon.daemon.utils import sleep_every
from travel.avia.ticket_daemon.ticket_daemon.lib.currency import Price
from travel.avia.ticket_daemon.ticket_daemon.lib.tracker import QueryTracker
from travel.avia.ticket_daemon.ticket_daemon.lib.partner_secret_storage import partner_secret_storage

log = getLogger(__name__)

BRAVOAVIA_URL = 'http://ws.viaggiare.it:80/vws/BravoFlySearchWs'
BRAVOAVIA_LOGIN = 'YANDEXWSRU'
BRAVOAVIA_PASSWORD = partner_secret_storage.get(
    importer_name='bravoavia', namespace='PASSWORD'
)


KLASS_MAP = {
    'economy': 'Y',
    'business': 'C',
    'first': 'F',
}


def validate_query(q):
    q.validate_klass(KLASS_MAP)


@QueryTracker.init_query
def query(tracker, q):
    xml = get_data(tracker, q)

    variants = parse_response(xml, q)

    return variants


def get_data(tracker, q):
    params = {
        'idBusinessProfile': BRAVOAVIA_LOGIN,
        'password': BRAVOAVIA_PASSWORD,
        'departureAirport': q.iata_from,
        'arrivalAirport': q.iata_to,
        'roundTrip': 'true' if q.date_backward else 'false',
        'outboundDate': q.date_forward.strftime("%Y-%m-%d"),
        'returnDate': q.date_backward.strftime("%Y-%m-%d") if q.date_backward else '',
        'adults': q.passengers.get('adults', 0),
        'childs': q.passengers.get('children', 0),
        'infants': q.passengers.get('infants', 0),
        'numberOfResults': 100,
        'language': {'ua': 'uk'}.get(q.national_version, q.national_version),
        'currency': 'UAH' if q.national_version == 'ua' else 'RUB',
        'classFlight': KLASS_MAP.get(q.klass, ''),
    }

    query_xml = loader.render_to_string(
        'partners/bravoavia.xml', params).strip()

    r = tracker.wrap_request(
        requests.post,
        BRAVOAVIA_URL,
        headers={
            'Content-Type': 'text/xml; charset=utf-8',
            'Connection': 'Keep-Alive',
            'SOAPAction': BRAVOAVIA_URL,
        },
        data=query_xml.encode('utf-8')
    )

    return r.content


def parse_response(xml, q):
    variants = []

    tree = etree.fromstring(xml)

    for index, e in sleep_every(enumerate(tree.xpath('//return/trips'))):

        v = Variant()

        amount = e.find('amount').text
        currency = e.find('currency').text

        v.tariff = Price(float(amount), {'RUB': 'RUR'}.get(currency, currency))

        v.klass = q.klass

        v.forward.segments = parse_flights(e.xpath('outboundLeg/hops'))

        if q.date_backward:
            v.backward.segments = parse_flights(e.xpath('returnLeg/hops'))

        v.url = e.find('deeplinkWithSharedData').text

        # RASPTICKETS-1613
        if q.national_version == 'ua':
            v.url = v.url.replace('www.bravoavia.ru',
                                  'www.bravoavia.com.ua', 1)

        v.order_data = {'url': v.url}

        variants.append(v)

    return variants


def parse_flights(segments):
    flights = []

    for seg in segments:
        f = IATAFlight()

        dep_datetime = seg.find('departureDate').text.split('+')[0]
        f.local_departure = datetime.strptime(
            dep_datetime, '%Y-%m-%dT%H:%M:%S')

        arr_datetime = seg.find('arrivalDate').text.split('+')[0]
        f.local_arrival = datetime.strptime(arr_datetime, '%Y-%m-%dT%H:%M:%S')

        f.station_from_iata = seg.find('departureAirport').text
        f.station_to_iata = seg.find('arrivalAirport').text

        f.company_iata = seg.find('idAirline').text

        f.number = '%s %s' % (
            f.company_iata,
            seg.find('flightNumber').text.lstrip('0'),
        )

        f.electronic_ticket = True

        flights.append(f)

    return flights
