# -*- coding: utf-8 -*-
from datetime import datetime
from logging import getLogger

import requests
from lxml import etree

from travel.avia.ticket_daemon.ticket_daemon.api.flights import Variant
from travel.avia.ticket_daemon.ticket_daemon.daemon.utils import sleep_every
from travel.avia.ticket_daemon.ticket_daemon.lib.baggage import BaggageParser
from travel.avia.ticket_daemon.ticket_daemon.lib.currency import Price
from travel.avia.ticket_daemon.ticket_daemon.lib.http import url_complement_missing
from travel.avia.ticket_daemon.ticket_daemon.lib.tracker import QueryTracker
from travel.avia.ticket_daemon.ticket_daemon.lib.partner_secret_storage import partner_secret_storage

log = getLogger(__name__)
baggage_parser = BaggageParser(log)

CHABOOKA_URL = 'http://api.chabooka.ru/search.php'
CHABOOKA_PARTNER = 'yandex'
CHABOOKA_PASSWORD = partner_secret_storage.get(
    importer_name='chabooka2', namespace='PASSWORD'
)

URL_TRACKER = {'utm_source': 'yandex-avia'}

KLASS_MAP = {'economy': 'Y', 'business': 'C'}


@QueryTracker.init_query
def query(tracker, q):
    # Партнер умеет отдавать только классы эконом и бизнес
    if q.klass not in KLASS_MAP.keys():
        return []

    xml = get_data(tracker, q)
    variants = list(parse_response(xml, q))

    return variants


def get_data(tracker, q):
    r = tracker.wrap_request(
        requests.post,
        CHABOOKA_URL,
        params=build_search_params(q)
    )

    return r.content


def build_search_params(q):
    params = {
        'from': q.iata_from,
        'to': q.iata_to,
        'date1': q.date_forward.strftime('%Y-%m-%d'),
        'adults': q.passengers.get('adults', 0),
        'children': q.passengers.get('children', 0),
        'infants': q.passengers.get('infants', 0),
        'partner': CHABOOKA_PARTNER,
        'password': CHABOOKA_PASSWORD,
        'cabin':  KLASS_MAP[q.klass],
        'precise': 1,
    }
    if q.date_backward:
        params['date2'] = q.date_backward.strftime('%Y-%m-%d')
    return params


def parse_response(xml, q):
    tree = etree.fromstring(xml)
    for e in sleep_every(tree.xpath('variant')):
        currency = e.find('currency').text

        if currency.lower() != 'rub':
            log.warning('Skip parsing (currency: %s)', currency)
            continue

        v = Variant()

        segments = e.xpath('segment')

        v.forward.segments = list(parse_flights(segments[0], q.importer.flight_fabric))
        has_backward = bool(q.date_backward and len(segments) > 1)
        if has_backward:
            v.backward.segments = list(parse_flights(segments[1], q.importer.flight_fabric))

        # # Пропускаем варианты с неправильными пересадками
        # if not v.forward.check_transfers_times() or \
        #         not v.backward.check_transfers_times():
        #     continue
        cabins = {ftag.find('cabin').text
                  for segment in segments[:2 if has_backward else 1] for ftag in segment}

        if len(cabins) > 1 or KLASS_MAP[q.klass] not in cabins:
            log.warning('Skip parsing (cabins: %s)', cabins)
            continue

        v.klass = q.klass

        v.tariff = Price(float(e.find('price').text), 'RUR')

        v.url = url_complement_missing(e.find('url').text, URL_TRACKER)
        v.order_data = {'url': v.url}

        v.charter = e.findtext('charter') == '1'

        yield v


def parse_flights(segment_tag, flight_fabric):
    for ftag in segment_tag.xpath('flight'):
        yield flight_fabric.create(
            station_from_iata=ftag.findtext('departure'),
            station_to_iata=ftag.findtext('arrival'),
            local_departure=datetime.strptime(
                ftag.findtext('departureDate') + ftag.findtext('departureTime'),
                '%Y-%m-%d%H:%M'),
            local_arrival=datetime.strptime(
                ftag.findtext('arrivalDate') + ftag.findtext('arrivalTime'),
                '%Y-%m-%d%H:%M'),
            company_iata=ftag.findtext('operatingCarrier'),
            pure_number=ftag.findtext('number'),
            baggage=baggage_parser.parse_from_string(ftag.findtext('baggage')),
        )
