# -*- coding: utf-8 -*-
import re
from datetime import datetime
from logging import getLogger
from urllib import urlencode

import requests
from lxml import etree

from travel.avia.ticket_daemon.ticket_daemon.api.flights import Variant
from travel.avia.ticket_daemon.ticket_daemon.daemon.utils import sleep_every
from travel.avia.ticket_daemon.ticket_daemon.lib.currency import Price
from travel.avia.ticket_daemon.ticket_daemon.lib.tracker import QueryTracker
from travel.avia.ticket_daemon.ticket_daemon.lib.partner_secret_storage import partner_secret_storage


log = getLogger(__name__)

CHARTERBILET_URL = 'http://charterbilet.ru/aviareg_railway/webservices/SOAPClient.php'
CHARTERBILET_LOGIN = 'wwwadm'
CHARTERBILET_PASSWD = partner_secret_storage.get(
    importer_name='charterbilet', namespace='PASSWORD'
)

KLASS_MAP = {'economy': u'1', 'business': u'3', 'first': u'2'}


@QueryTracker.init_query
def query(tracker, q):
    xml = get_data(tracker, q)

    variants = list(parse_response(xml, q))

    return variants


def build_search_params(q):
    return {
        'available': 'Y',
        'from': q.iata_from.encode('utf-8'),
        'to': q.iata_to.encode('utf-8'),
        'date1': q.date_forward.strftime('%d.%m.%Y'),
        'date2': q.date_backward and q.date_backward.strftime('%d.%m.%Y') or '',
        'oneway': '' if q.date_backward else 'Y',
        'adult_seats': q.passengers.get('adults', 0),
        'children_seats': q.passengers.get('children', 0),
        'infanf_seats': q.passengers.get('infants', 0),
        'class': KLASS_MAP.get(q.klass, u'1'),
        'flex_dates': 'no',
    }


def get_data(tracker, q):
    r = tracker.wrap_request(
        requests.post,
        CHARTERBILET_URL,
        headers={
            'Content-Type': 'application/x-www-form-urlencoded',
        },
        auth=requests.auth.HTTPBasicAuth(
            CHARTERBILET_LOGIN,
            CHARTERBILET_PASSWD,
        ),
        data=urlencode(build_search_params(q))
    )

    return r.content


def parse_response(xml, q):
    # Выкидываем php нотисы из ответа, оставляем только root
    xml = re.search(r'<FlightsSearchRS>.*</FlightsSearchRS>', xml, re.M | re.U)

    if xml:
        xml = xml.group(0)
    else:
        return

    tree = etree.fromstring(xml)

    for e in sleep_every(tree.xpath('/FlightsSearchRS/Variant')):
        v = Variant()

        v.tariff = Price(float(e.find('TotalPrice').text))

        segments_out = e.xpath(
            'FlightRoute/Flights[@RouteType="Forward" or @RouteType="Next"]')

        segments_in = e.xpath('FlightRoute/Flights[@RouteType="Back"]')

        if segments_out:
            v.forward.segments = list(parse_flights(
                segments_out[0].xpath('Flight'), q.importer.flight_fabric))

        if not v.forward.segments:
            continue

        if v.forward.segments[0].local_departure.date() != q.date_forward:
            continue

        if segments_in:
            v.backward.segments = list(parse_flights(
                segments_in[0].xpath('Flight'), q.importer.flight_fabric))

            if q.date_backward and v.backward.segments and \
               v.backward.segments[0].local_departure.date() != q.date_backward:
                continue

        v.url = e.get('URL')

        v.klass = q.klass

        v.order_data = {
            'url': v.url,
        }

        v.charter = True

        yield v


def parse_flights(segments, flight_fabric):
    for segment in segments:
        dep = segment.find('Departure')
        arr = segment.find('Arrival')
        elem_carrier = segment.find('Carrier')

        _company_iata = (
            elem_carrier.attrib.get('Code_IATA') or
            elem_carrier.attrib.get('Code_ICAO') or
            elem_carrier.text)
        yield flight_fabric.create(
            station_from_iata=dep.find('Airport').text,
            station_to_iata=arr.find('Airport').text,
            local_departure=datetime.strptime(
                dep.get('Date') + dep.get('Time'), '%d.%m.%y%H:%M'),
            local_arrival=datetime.strptime(
                arr.get('Date') + arr.get('Time'), '%d.%m.%y%H:%M'),
            company_iata=_company_iata,
            pure_number=segment.get('FlightNumber'),
        )
