# -*- coding: utf-8 -*-
from datetime import datetime, timedelta
from itertools import product
from logging import getLogger

import requests
from lxml import etree

from travel.avia.ticket_daemon.ticket_daemon.api.flights import Variant
from travel.avia.ticket_daemon.ticket_daemon.daemon.utils import sleep_every
from travel.avia.ticket_daemon.ticket_daemon.lib.currency import Price
from travel.avia.ticket_daemon.ticket_daemon.lib.http import url_complement_missing
from travel.avia.ticket_daemon.ticket_daemon.lib.tracker import QueryTracker
from travel.avia.ticket_daemon.ticket_daemon.lib.partner_secret_storage import partner_secret_storage


log = getLogger(__name__)

SEARCH_URL = 'http://gate.ru-charter.ru/xml.asp'
CLIENT_ID = 545
KEY = partner_secret_storage.get(importer_name='charteok', namespace='PASSWORD')
BOOK_REDIRECT_URL = 'https://www.charterok.ru/Passengers.aspx'
URL_TRACKER = {'utm_source': 'yandex.avia'}

KLASS_MAP = {'Y': 'economy', 'C': 'business', 'F': 'first'}


@QueryTracker.init_query
def query(tracker, q):
    xml = get_data(tracker, q)
    variants = parse_response(xml, q)

    return variants


def validate_query(q):
    q.validate_klass(['economy'])


def get_data(tracker, q):
    params = build_search_params(q)

    r = tracker.wrap_request(
        requests.post,
        SEARCH_URL,
        headers={
            'Content-Type': 'application/x-www-form-urlencoded; charset=UTF-8',
        },
        data=build_query_xml(params)
    )

    return r.content


def build_search_params(q):
    passengers_count = \
        q.passengers.get('adults', 0) + q.passengers.get('children', 0)

    params = {
        'type': 'getTariffList',
        'city_from': q.iata_from,
        'city_to': q.iata_to,
        'date_from': q.date_forward.strftime('%d.%m.%Y'),
        'tariff_type': 'RT' if q.date_backward else 'OW',
        'range': '0',
        'is_no_range': '0',
        'tariff_month': '',
        'is_tariff_range': '0',
        'date_range_from': '',
        'date_range_to': '',
        'range_duration': '1',
        'range_value': '0',
        'is_seats': '0',
        'pass_count': passengers_count,
        'rec_count': '0',
        'page_number': '0',
    }

    if q.date_backward:
        params['date_to'] = q.date_backward.strftime('%d.%m.%Y')

    return params


def build_query_xml(params):
    data = etree.Element('data')
    request = etree.SubElement(data, 'request')

    for key, val in params.items():
        request.set(unicode(key), unicode(val))

    client = etree.SubElement(data, 'client')
    etree.SubElement(client, 'clientId').text = str(CLIENT_ID)
    etree.SubElement(client, 'key').text = str(KEY)
    etree.SubElement(client, 'token').text = ''

    return etree.tostring(
        data,
        encoding='utf-8',
        xml_declaration=True
    )


def parse_response(xml, q):
    tree = etree.fromstring(xml)
    variants = []

    for tariff_tag in sleep_every(tree.xpath('/data/tariffList/tariff')):

        variants_tags = tariff_tag.xpath('legs/leg[@dir="0"]//variant')

        forward_variants = [
            (v_tag, list(parse_flights(v_tag.xpath('flight'), q.importer.flight_fabric)))
            for v_tag in variants_tags
        ]

        if q.date_backward:
            variants_tags = tariff_tag.xpath('legs/leg[@dir="1"]//variant')

            if not variants_tags:
                raise Exception('No backward variants in RT search result')

            backward_variants = [
                (v_tag, list(parse_flights(v_tag.xpath('flight'), q.importer.flight_fabric)))
                for v_tag in variants_tags
            ]

        else:
            backward_variants = [None]

        for forward, backward in product(forward_variants, backward_variants):
            v = Variant()

            forward_v_tag, v.forward.segments = forward

            if backward:
                backward_v_tag, v.backward.segments = backward
                # id обратного перелёта в приоритете
                variant_id = backward_v_tag.get('id')

            else:
                variant_id = forward_v_tag.get('id')

            # <price adt="9370.0000" chd="9370.0000" inf="2706.0000" currency="RUR"/>
            price_tag = tariff_tag.find('price')

            price_val = (
                float(price_tag.get('adt')) * q.passengers.get('adults', 0) +
                float(price_tag.get('chd')) * q.passengers.get('children', 0) +
                float(price_tag.get('inf')) * q.passengers.get('infants', 0)
            )

            v.tariff = Price(price_val, price_tag.get('currency'))

            v.klass = KLASS_MAP[tariff_tag.get('class')]

            v.charter = any([f.charter for f in v.all_segments])

            v.order_data = {
                'variant': variant_id,
                'adults': q.passengers.get('adults', 0),
                'children': q.passengers.get('children', 0),
                'infants': q.passengers.get('infants', 0),
                'tariff_type': 'RT' if q.date_backward else 'OW'
            }

            variants.append(v)

    return variants


def parse_flights(f_tags, flight_fabric):
    for f_tag in f_tags:
        local_arrival = datetime.strptime(
            '%s %s' % (f_tag.get('date'), f_tag.get('arr')),
            '%Y-%m-%d %H:%M'
        )
        # признак прилета на следующие сутки (0 – нет, 1 – да)
        if int(f_tag.get('arrflag', 0)):
            local_arrival += timedelta(1)
        yield flight_fabric.create(
            station_from_iata=f_tag.get('from') or f_tag.get('city_from'),
            station_to_iata=f_tag.get('to') or f_tag.get('city_to'),
            local_departure=datetime.strptime(
                '%s %s' % (f_tag.get('date'), f_tag.get('dep')),
                '%Y-%m-%d %H:%M'
            ),
            local_arrival=local_arrival,
            company_iata=f_tag.get('airline'),
            pure_number=f_tag.get('flnum'),
            charter=f_tag.get('charter') == 'true',
            )


def book(order_data):
    postdata = {
        'meta_id': CLIENT_ID,
        'meta_guid': KEY,
        'variant': order_data['variant'],
        'adult': order_data['adults'],
        'child': order_data['children'],
        'infant': order_data['infants'],
        'tariff_type': order_data['tariff_type'],
    }

    return url_complement_missing(BOOK_REDIRECT_URL, URL_TRACKER), postdata
