# -*- coding: utf-8 -*-
from datetime import datetime

import requests
from lxml import etree

from travel.avia.ticket_daemon.ticket_daemon.api.flights import IATAFlight, Variant
from travel.avia.ticket_daemon.ticket_daemon.daemon.utils import sleep_every
from travel.avia.ticket_daemon.ticket_daemon.lib.currency import Price
from travel.avia.ticket_daemon.ticket_daemon.lib.decorators import elementwise
from travel.avia.ticket_daemon.ticket_daemon.lib.tracker import QueryTracker
from travel.avia.ticket_daemon.ticket_daemon.lib.utils import skip_None_values
from travel.avia.ticket_daemon.ticket_daemon.lib.partner_secret_storage import partner_secret_storage


def flip(d):
    ks, vs = zip(*(d.items()))
    return dict(zip(vs, ks))


KLASS_MAP = {
    'economy': 'E',
}
SERVICE_CLASSES = flip(KLASS_MAP)


def validate_query(q):
    q.validate_klass(KLASS_MAP)


@QueryTracker.init_query
def query(tracker, q):
    xml = get_data(tracker, q)
    return list(iparse_response(xml, q))


def get_data(tracker, q):
    r = tracker.wrap_request(
        requests.get,
        'http://chartertickets.com.ua/api.php',
        params=skip_None_values({
            'token': partner_secret_storage.get(
                importer_name='chartertickets', namespace='PASSWORD'
            ),
            'departure': q.iata_from,
            'arrival': q.iata_to,
            'date_forward': q.date_forward.strftime('%Y-%m-%d'),
            'date_backward': q.date_backward and q.date_backward.strftime('%Y-%m-%d') or None,
            'class': KLASS_MAP[q.klass],
            'adult': q.passengers.get('adults', 0),
            'child': q.passengers.get('children', 0),
            'infant': q.passengers.get('infants', 0),
        }),
        headers={
            # Отвечают 424 (Failed Dependency) если "User-Agent": "python-requests
            'User-Agent': 'Yandex avia service',
        }
    )
    return r.content


def iparse_response(xml, q):
    tree = etree.fromstring(xml)

    for variant_tag in sleep_every(tree.xpath('variant')):

        v = Variant()

        fare_tag = variant_tag.find('fare')
        v.tariff = Price(float(fare_tag.get('value')), fare_tag.get('currency'))
        v.charter = fare_tag.get('charter') == 'true'
        v.klass = SERVICE_CLASSES.get(fare_tag.get('class'))

        v.forward.segments = parse_flights(variant_tag.xpath('route_forward'))

        if q.date_backward:
            v.backward.segments = parse_flights(variant_tag.xpath('route_backward'))

        v.url = variant_tag.get('url')
        v.order_data = {'url': v.url}

        yield v


@elementwise
def parse_flights(f_tag):
    f = IATAFlight()

    f.station_from_iata = f_tag.get('departure_airport_code')
    f.station_to_iata = f_tag.get('arrival_airport_code')

    f.local_departure = pwhen(f_tag.get('departure_datetime'))
    f.local_arrival = pwhen(f_tag.get('arrival_datetime'))

    f.company_iata = f_tag.get('company_code')
    f.number = f.company_iata + ' ' + f_tag.get('route_code')

    f.fare_code = f_tag.get('farecode')

    return f


def pwhen(val):
    return datetime.strptime(val, '%Y-%m-%d %H:%M')
