# -*- coding: utf-8 -*-
from datetime import datetime, timedelta
from logging import getLogger
from urllib import urlencode

import requests
from lxml import etree

from travel.avia.ticket_daemon.ticket_daemon.api.flights import Variant
from travel.avia.ticket_daemon.ticket_daemon.api.query import QueryIsNotValid
from travel.avia.ticket_daemon.ticket_daemon.daemon.utils import (
    async_sleep, collect_workers_results, spawn_worker
)
from travel.avia.ticket_daemon.ticket_daemon.lib.currency import Price
from travel.avia.ticket_daemon.ticket_daemon.lib.tracker import QueryTracker
from travel.avia.ticket_daemon.ticket_daemon.api.models_utils.search_codes import get_chartex_city_code_by_point

log = getLogger(__name__)

CHARTEX_URL = 'http://xml.chartex.ru/gate/gate.asp'
CHARTEX_UID = '61500874'
CHARTEX_ORDER_URL = 'http://www.chartex.ru/step2.asp'

RECORDS_ON_PAGE = 100


def validate_query(q):
    q.validate_klass(['economy'])

    log.debug('validate query: %r', q.id)

    q.chartex_city_code_from = get_chartex_city_code_by_point(q.point_from)

    log.debug('q.chartex_city_code_from: %r', q.chartex_city_code_from)
    if not q.chartex_city_code_from:
        raise QueryIsNotValid('No chartex city code: %r', q.point_from)

    q.chartex_city_code_to = get_chartex_city_code_by_point(q.point_to)

    log.debug('q.chartex_city_code_to: %r', q.chartex_city_code_to)
    if not q.chartex_city_code_to:
        raise QueryIsNotValid('No chartex city code: %r', q.point_to)


@QueryTracker.init_query
def query(tracker, q):
    variants = []

    pages_count = 1
    page_num = 0

    while page_num < pages_count:
        async_sleep(0)

        page_num += 1

        params = build_query_charters_page_params(q, page_num)

        charters_xml = get_data(tracker, q, params)

        try:
            charter_tags, pages_count = parse_charters_xml(charters_xml)

        except Exception:
            log.warning(u'Bad query charters page result %s', repr(params))

            continue

        charter_xmls = collect_workers_results([
            spawn_worker(
                get_data,
                tracker, q, build_query_charter_params(charter_tag, q)
            )
            for charter_tag in charter_tags
        ])

        page_variants = filter(None, [
            parse_variant(xml, charter_tag, q)
            for xml, charter_tag in zip(charter_xmls, charter_tags)
        ])

        log.debug(u'Page %s variants count: %s', page_num, len(page_variants))

        variants.extend(page_variants)

    log.debug(u'All variants count: %s', len(variants))

    return variants


def build_query_charters_page_params(q, page_num):
    params = {
        'request': 'step1rub',
        'UID': CHARTEX_UID,
        'CityFromID': q.chartex_city_code_from.encode('utf-8'),
        'CityToID': q.chartex_city_code_to.encode('utf-8'),
        'Tariff': 2 if q.date_backward else 1,
        'DateDep': q.date_forward.strftime('%d.%m.%Y'),
        'Page': page_num
    }

    if q.date_backward:
        params['DateRet'] = q.date_backward.strftime('%d.%m.%Y')

    return params


def build_query_charter_params(charter_tag, q):
    return {
        'request': 'reisinfo',
        'UID': CHARTEX_UID,
        'TID': charter_tag.get('id'),
        'DateFrom': charter_tag.get('date', ''),
        'DateBack': charter_tag.get('dateBack', '')
    }


def get_data(tracker, q, params):
    r = tracker.wrap_request(
        requests.post,
        CHARTEX_URL,
        headers={
            'Content-Type':
                'application/x-www-form-urlencoded; charset=windows-1251',
        },
        data=urlencode(params)
    )

    return r.content


def parse_charters_xml(xml):
    async_sleep(0)

    charter_tree = etree.fromstring(xml)

    charter_tags = charter_tree.xpath('charter')

    city = charter_tree.find('city')

    records_count = int(city.get('reccount'))

    log.debug(u'Charters xml records count: %s', records_count)

    pages_count = (records_count + 1) / RECORDS_ON_PAGE + 1

    return charter_tags, pages_count


def parse_variant(xml, charter_tag, q):
    flight_tree = etree.fromstring(xml)

    cost_adults = float(charter_tag.get('price')) * \
        float(q.passengers.get('adults', 1))

    cost_children = float(charter_tag.get('priceChild')) * \
        float(q.passengers.get('children', 0))

    cost_infants = float(charter_tag.get('priceInf')) * \
        float(q.passengers.get('infants', 0))

    v = Variant()

    v.tariff = Price(float(cost_adults + cost_children + cost_infants), 'RUR')

    v.klass = 'economy'

    v.url = build_order_url(q, charter_tag)
    v.order_data = {'url': v.url}

    v.forward.segments = parse_flights(
        flight_tree[0],
        charter_tag.get('date'),
        q.importer.flight_fabric,
    )

    v.charter = True

    if q.date_backward:
        v.backward.segments = parse_flights(
            flight_tree[1],
            charter_tag.get('dateBack'),
            q.importer.flight_fabric,
        )

        if not v.backward.segments:
            return None

    return v

BAD_TIME_FORMATS = ['', 'xx:xx']


def parse_datetime(date_str, time_str, next_day_arrival=False):
    if time_str in BAD_TIME_FORMATS:
        return None

    try:
        dt = datetime.strptime(
            date_str + ' ' + time_str,
            '%d.%m.%Y %H:%M'
        )
        if next_day_arrival:
            dt += timedelta(days=1)
        return dt
    except ValueError:
        return None


def parse_flights(flight_tree, flight_date, flight_fabric):
    flights = []

    flight_code = flight_tree.get('code')

    _company_iata = flight_code.split()[0]
    departure_time = flight_tree.get('deptime')
    arrival_time = flight_tree.get('arrtime')
    f = flight_fabric.create(
        company_iata=_company_iata,
        number=flight_code,
        station_from_iata=flight_tree.get('portFrom')[:3],
        station_to_iata=flight_tree.get('portTo')[:3],
        local_departure=parse_datetime(flight_date, departure_time),
        local_arrival=parse_datetime(
            flight_date, arrival_time,
            next_day_arrival=arrival_time < departure_time
        ),
    )

    if f.local_departure and f.local_arrival:
        flights.append(f)

    return flights


def build_order_url(q, charter_tag):
    params = {
        'CityFromID': q.chartex_city_code_from.encode('utf-8'),
        'CityToID': q.chartex_city_code_to.encode('utf-8'),
        'sTariff': 2 if q.date_backward else 1,
        'Adult': q.passengers.get('adults', 0),
        'Child': q.passengers.get('children', 0),
        'Inf': q.passengers.get('infants', 0),
        'TID': charter_tag.get('id'),
        'DateFrom': charter_tag.get('date'),
        'InterfaceID': 32,
    }

    if q.date_backward:
        params['DateBack'] = charter_tag.get('dateBack')

    return CHARTEX_ORDER_URL + '?' + urlencode(params)
