# -*- coding: utf-8 -*-
from collections import namedtuple

from datetime import datetime
from logging import getLogger

import requests
from lxml import etree

from travel.avia.ticket_daemon.ticket_daemon.api.flights import Variant
from travel.avia.ticket_daemon.ticket_daemon.daemon.utils import sleep_every
from travel.avia.ticket_daemon.ticket_daemon.lib.baggage import Baggage
from travel.avia.ticket_daemon.ticket_daemon.lib.currency import Price
from travel.avia.ticket_daemon.ticket_daemon.lib.tracker import QueryTracker
from travel.avia.ticket_daemon.ticket_daemon.lib.partner_secret_storage import partner_secret_storage

log = getLogger(__name__)


Auth = namedtuple('Auth', ('partner', 'password'))

SEARCH_URL = 'http://avia.city.travel/aviasales/search'
AUTH_BY_NV = {
    'ru': Auth('yandex', partner_secret_storage.get('citytravel', 'PASSWORD')),
    'kz': Auth('yandexkz', partner_secret_storage.get('citytravel_kz', 'PASSWORD')),
}


@QueryTracker.init_query
def query(tracker, q):
    xml = get_data(tracker, q)
    variants = parse_response(xml, q)

    return variants


def get_data(tracker, q):
    auth = AUTH_BY_NV.get(q.national_version) or AUTH_BY_NV['ru']
    params = {
        'from': q.iata_from,
        'to': q.iata_to,
        'date1': q.date_forward.strftime('%Y-%m-%d'),
        'cabin': {'economy': 'Y', 'business': 'B'}[q.klass],
        'adults': q.passengers.get('adults', 0),
        'children': q.passengers.get('children', 0),
        'infants': q.passengers.get('infants', 0),
        'partner': auth.partner,
        'password': auth.password,
        'gzip': 1,
    }

    if q.date_backward:
        params['date2'] = q.date_backward.strftime('%Y-%m-%d')

    r = tracker.wrap_request(
        requests.get,
        SEARCH_URL,
        params=params
    )

    return r.content


SERVICE_CLASS_MAP = {
    'Y': 'economy',
    'C': 'business',
    'F': 'first',
}


def parse_response(xml, q):
    variants = []
    tree = etree.fromstring(xml)

    for v_tag in sleep_every(tree.xpath('variant')):
        v = Variant()

        carrier = v_tag.findtext('validatingCarrier')
        segments = v_tag.xpath('segment')

        v.forward.segments = list(parse_flights(
            q.importer.flight_fabric,
            segments[0].xpath('flight'), carrier
        ))

        if q.date_backward and len(segments) > 1:
            v.backward.segments = list(parse_flights(
                q.importer.flight_fabric,
                segments[1].xpath('flight'), carrier
            ))

        price_val = float(v_tag.findtext('price'))
        currency = v_tag.findtext('currency')

        if currency == 'RUB':
            currency = 'RUR'

        v.tariff = Price(price_val, currency)

        v.klass = q.klass

        v.url = v_tag.findtext('url')

        v.order_data = {'url': v.url}

        v.charter = any(s.charter for s in v.iter_all_segments())

        variants.append(v)

    return variants


def extract_datetime(tag, date_attr, time_attr):
    raw_date = tag.findtext(date_attr)
    raw_time = tag.findtext(time_attr)

    return datetime(
        int(raw_date[0:4]), int(raw_date[5:7]), int(raw_date[8:10]),
        int(raw_time[0:2]), int(raw_time[3:5]),
    )


def _get_element_if_exist(tag, element_path):
    result_element = tag.findtext(element_path)
    if result_element is not None:
        return int(result_element)
    return None


def get_baggage(f_tag):
    included = f_tag.findtext('luggage')

    if included == 'false':
        return Baggage.from_partner(included=False)
    elif included == 'true':
        return Baggage.from_partner(
            included=True,
            pieces=_get_element_if_exist(f_tag, 'pieces_of_luggage'),
            weight=_get_element_if_exist(f_tag, 'luggage_weight'),
        )

    return Baggage.from_partner()


def parse_flights(flight_fabric, flights_tags, carrier):
    for f_tag in flights_tags:
        company_iata = f_tag.findtext('operatingCarrier')
        if company_iata is None:
            company_iata = carrier

        yield flight_fabric.create(
            station_from_iata=f_tag.findtext('departure'),
            station_to_iata=f_tag.findtext('arrival'),
            local_departure=extract_datetime(f_tag, 'departureDate', 'departureTime'),
            local_arrival=extract_datetime(f_tag, 'arrivalDate', 'arrivalTime'),
            klass=SERVICE_CLASS_MAP[f_tag.findtext('cabin')],
            company_iata=company_iata,
            pure_number=f_tag.findtext('number'),
            charter=f_tag.findtext('charter') == '1',
            selfconnect=f_tag.findtext('selfconnect') == '1',
            fare_code=f_tag.findtext('FareBasis'),
            baggage=get_baggage(f_tag),
        )
