# -*- coding: utf-8 -*-
import itertools
from datetime import datetime
from logging import getLogger

import requests
from lxml import etree

from travel.avia.ticket_daemon.ticket_daemon.api.flights import Variant
from travel.avia.ticket_daemon.ticket_daemon.daemon.utils import sleep_every
from travel.avia.ticket_daemon.ticket_daemon.lib.currency import Price
from travel.avia.ticket_daemon.ticket_daemon.lib.decorators import pipe
from travel.avia.ticket_daemon.ticket_daemon.lib.tracker import QueryTracker
from travel.avia.ticket_daemon.ticket_daemon.lib.baggage import BaggageParser

log = getLogger(__name__)
baggage_parser = BaggageParser(logger=log)

CLICKAVIA_URL = 'http://www.clickavia.ru/search/flights'


@QueryTracker.init_query
def query(tracker, q):
    # Будем спрашивать только классы эконом или бизнес
    if q.klass not in ['economy', 'business']:
        return []

    xml = get_data(tracker, q)

    variants = parse_response(xml, q)

    return variants


def get_data(tracker, q):
    params = [
        ('depart_city', q.iata_from),
        ('destin_city', q.iata_to),
        ('depart_date', q.date_forward.strftime('%d.%m.%Y')),
        ('return_date', q.date_backward and q.date_backward.strftime('%d.%m.%Y') or ''),
        ('seats_class', {'economy': u'0', 'business': u'1'}[q.klass]),
        ('range', 0),
        ('referer', 'yandex'),
        ('one_way', 0 if q.date_backward else 1),
        ('result_type', 'xml'),
        ('get_charts', 0),
        ('get_all_flights', 1),
        ('adult', q.passengers.get('adults', 0)),
        ('children', q.passengers.get('children', 0)),
        ('infant', q.passengers.get('infants', 0)),
    ]

    r = tracker.wrap_request(
        requests.get,
        CLICKAVIA_URL,
        params=params
    )

    return r.content


@pipe(list)
def parse_response(xml, q):
    tree = etree.fromstring(xml)

    for e in sleep_every(tree.xpath('/result/flights/trip')):

        v = Variant()

        v.tariff = Price(float(e.findtext('price')))

        v.forward.segments = parse_flights(
            e.xpath('flight_to/segments/segment'),
            q.importer.flight_fabric,
            e.findtext('flight_to/self_connect') == '1'
        )

        if not v.forward.segments:
            continue

        v.backward.segments = parse_flights(
            e.xpath('flight_return/segments/segment'),
            q.importer.flight_fabric,
            e.findtext('flight_return/self_connect') == '1'
        )

        v.url = e.findtext('link')

        v.klass = q.klass

        v.order_data = {
            'url': v.url,
        }

        v.charter = any(s.charter for s in itertools.chain(v.forward.segments, v.backward.segments))

        yield v


@pipe(list)
def parse_flights(segments, flight_fabric, selfconnect):
    for segment in segments:
        number = segment.findtext('code')

        yield flight_fabric.create(
            selfconnect=selfconnect,
            station_from_iata=segment.findtext('depart_airport')[:3],
            station_to_iata=segment.findtext('arrive_airport')[:3],
            local_departure=datetime.strptime(
                segment.findtext('date') + segment.findtext('depart_time'),
                '%Y-%m-%d%H:%M'
            ),
            local_arrival=datetime.strptime(
                segment.findtext('arrive_date') + segment.findtext('arrive_time'),
                '%Y-%m-%d%H:%M'
            ),
            company_iata=number.split()[0],
            number=number,
            baggage=baggage_parser.parse_from_string(segment.findtext('baggage')),
            charter=segment.findtext('type') == 'charter'
        )
