# -*- coding: utf-8 -*-
import urllib
from datetime import datetime
from logging import getLogger

import requests
from lxml import etree

from travel.avia.ticket_daemon.ticket_daemon.api.flights import Variant
from travel.avia.ticket_daemon.ticket_daemon.api.query import QueryIsNotValid
from travel.avia.ticket_daemon.ticket_daemon.daemon.utils import BadPartnerResponse, sleep_every
from travel.avia.ticket_daemon.ticket_daemon.lib.currency import Price
from travel.avia.ticket_daemon.ticket_daemon.lib.decorators import pipe
from travel.avia.ticket_daemon.ticket_daemon.lib.http import update_query_string
from travel.avia.ticket_daemon.ticket_daemon.lib.tracker import QueryTracker
from travel.avia.ticket_daemon.ticket_daemon.lib.utils import skip_None_values
from travel.avia.ticket_daemon.ticket_daemon.lib.partner_secret_storage import partner_secret_storage

log = getLogger(__name__)

SEARCH_URL = 'https://mse.csa.cz/en/flightItinerary.php'
LOGIN = 'mse_yandex'
PASSWORD = 'csa_Ya761_789'
WEBID = 'yandex'
KLASS_MAP = {'economy': 'Y', 'business': 'C'}


def validate_query(q):
    if not all([hasattr(q, 'iata_real_to'), hasattr(q, 'iata_real_from')]):
        raise QueryIsNotValid('Only IATA codes allowed: %s %s' % (
            q.iata_from.encode('utf-8'),
            q.iata_to.encode('utf-8')
        ))


@QueryTracker.init_query
def query(tracker, q):
    tree = get_data(tracker, q)
    variants = parse_response(tree, q)

    return variants


def build_search_params(q):
    return skip_None_values({
        'version': '1-0',
        'webId': WEBID,
        'user': LOGIN,
        'password': partner_secret_storage.get(
            importer_name='csa', namespace='PASSWORD'
        ),
        'pos': 'RU',
        'dep': q.iata_from.encode('utf-8'),
        'arr': q.iata_to.encode('utf-8'),
        'departure': q.date_forward.strftime('%Y-%m-%d'),
        'return': q.date_backward and q.date_backward.strftime('%Y-%m-%d') or None,
        'class': KLASS_MAP[q.klass],
        'adtcount': q.passengers.get('adults', 0),
        'chdcount': q.passengers.get('children', 0),
        'infcount': q.passengers.get('infants', 0),
    })


def get_data(tracker, q):
    r = tracker.wrap_request(
        requests.get,
        update_query_string(
            SEARCH_URL,
            build_search_params(q)
        )
    )

    tree = etree.fromstring(r.content)
    error = tree.xpath('//data/error')

    if error:
        raise BadPartnerResponse('csa', r)

    return tree


def parse_response(tree, q):
    variants = []

    data = tree.xpath('//data')[0]
    deeplink = data.findtext('deeplink')

    for x, e in sleep_every(enumerate(data.xpath('itineraryList/itinerary'))):
        v = Variant()

        v.forward.segments = parse_segments(
            e.xpath('departure/segmentList/segment'), q.importer.flight_fabric
        )

        if q.date_backward:
            v.backward.segments = parse_segments(
                e.xpath('return/segmentList/segment'), q.importer.flight_fabric
            )

        v.tariff = Price(float(e.find('price/totalAmount').text))

        v.klass = q.klass

        v.url = urllib.unquote(deeplink)

        v.order_data = {'url': v.url}

        # Умеют возвращать XML с отсуствующими сегментами в XML
        if v.forward.segments:
            variants.append(v)

    return variants


DT_FMT = '%Y-%m-%dT%H:%M'
DT_LEN = len(datetime.now().strftime(DT_FMT))


@pipe(list)
def parse_segments(items, flight_fabric):
    for item in items:
        operatingCarrier = item.find('operatingCarrier')
        yield flight_fabric.create(
            station_from_iata=item.findtext('departureAirport'),
            station_to_iata=item.findtext('arrivalAirport'),
            local_departure=datetime.strptime(
                item.findtext('departureTime')[:DT_LEN], DT_FMT
            ),
            local_arrival=datetime.strptime(
                item.findtext('arrivalTime')[:DT_LEN], DT_FMT
            ),
            company_iata=operatingCarrier.text or 'OK',
            pure_number=item.findtext('flightNumber'),
            # electronic_ticket=item.findtext('ticketType') == 'ET'
        )
