# -*- coding: utf-8 -*-
import logging
from collections import namedtuple
from datetime import datetime, timedelta
from functools import partial
from uuid import uuid4

import requests
import ujson
from django.conf import settings

from travel.avia.ticket_daemon.ticket_daemon.api.flights import Variant
from travel.avia.ticket_daemon.ticket_daemon.api.cache import shared_cache
from travel.avia.library.python.ticket_daemon.memo import CacheInMemcache, memoize
from travel.avia.ticket_daemon.ticket_daemon.api.query import QueryIsNotValid
from travel.avia.ticket_daemon.ticket_daemon.daemon.utils import sleep_every
from travel.avia.ticket_daemon.ticket_daemon.lib.baggage import Baggage
from travel.avia.ticket_daemon.ticket_daemon.lib.currency import Price
from travel.avia.ticket_daemon.ticket_daemon.lib.http import url_complement_missing
from travel.avia.ticket_daemon.ticket_daemon.lib.partner_secret_storage import partner_secret_storage
from travel.avia.ticket_daemon.ticket_daemon.lib.tracker import QueryTracker

log = logging.getLogger(__name__)

SERVICE_URL = 'https://sopenservice.ctrip.com/OpenService/ServiceProxy.ashx'
AUTH_URL = 'https://openserviceauth.ctrip.com/OpenServiceAuth/authorize.ashx'

Config = namedtuple('Config', ('aid', 'sid', 'sid_name', 'api_key'))
AID = 844133
AID_NAME = 'Yandex Flights'

get_secret = partial(partner_secret_storage.get, namespace='PASSWORD')
CONFIGS_BY_NV = {
    'ru': Config(AID, 1409540, 'Yandex Flights_RU', get_secret(importer_name='ctrip_ru')),
    'kz': Config(AID, 1409795, 'Yandex Flights_KZ', get_secret(importer_name='ctrip_kz')),
    'tr': Config(AID, 1409794, 'Yandex Flights_TR', get_secret(importer_name='ctrip_tr')),
    'ua': Config(AID, 1409792, 'Yandex Flights_UA', get_secret(importer_name='ctrip_ua')),
    'com': Config(AID, 1409797, 'Yandex Flights_Global', get_secret(importer_name='ctrip_com')),
}

KLASS_MAP = {'economy': 'Economy', 'business': 'Business', 'first': 'First'}


@memoize(lambda tracker, config: config.api_key, CacheInMemcache(
    shared_cache, 480, prefix='ctrip', suffix='auth',
))
def _access_token(tracker, config):
    auth_response = tracker.wrap_request(
        requests.get,
        AUTH_URL,
        params={'AID': config.aid, 'SID': config.sid, 'KEY': config.api_key},
    )
    return auth_response.json()['Access_Token']


def _search_segment(iata_from, iata_to, dt):
    return {
        'DCityCode': iata_from,
        'ACityCode': iata_to,
        'DDate': dt.strftime('%Y-%m-%d'),
    }


def request_json(q):
    search_segments = [_search_segment(q.iata_from, q.iata_to, q.date_forward)]
    currency = 'RUB' if q.national_version == 'ru' else settings.AVIA_NATIONAL_CURRENCIES[
        q.national_version]
    country = q.national_version.upper()
    country = country if country != 'COM' else 'DE'

    if q.date_backward:
        search_segments.append(
            _search_segment(q.iata_to, q.iata_from, q.date_backward)
        )

    num_adults = q.passengers.get('adults', 1)
    num_children = q.passengers.get('children', 0)
    num_infants = q.passengers.get('infants', 0)

    if num_adults > 1:
        num_children = 0
        num_infants = 0

    return {
        'TravelerNumber': {
            'Adult': num_adults,
            'Child': num_children,
            'Infant': num_infants,
            'TravelerEligibilityCodeType': 'ALL',
        },

        'SegmentParameterList': search_segments,

        'Head': {
            'APIKey': CONFIGS_BY_NV[q.national_version].api_key,
            'Currency': currency,
            'Language': q.lang.upper(),
            'Site': _get_site_version(q.national_version),
            'Country': country,
        },
        'IsShowAll': 0,
        'PartitionedToken': '',
        'SearchType': 'Asyn',
        'TripType': 'RT' if q.date_backward else 'OW',
        'ClassType': KLASS_MAP[q.klass],
        'IsCombination': 1,
        'IsLowPrice': 1,
        'MaxDepartCount': 500,

        'SequenceInfo': {
            'SequenceType': 'Price',
            'CollationsType': 'Asc',
        },
        'FilterInfo': {
            'TransitType': '',
            'AirLine': '',
            'DPort': '',
            'APort': '',
            'DDateStr': 0,
            'DDateEnd': 0,
            'ADateStr': 0,
            'ADateEnd': 0,
            'GaTravelerEligibilityCodeType': '',
        },
        'ProductKeyInfo': {
            'No': '',
            'PartitionedToken': '',
            'NextGroupKey': '',
            'PlanCategoryName': '',
        },
    }


@QueryTracker.init_query
def query(tracker, q):
    return parse_response(get_data(tracker, q), q)


def validate_query(q):
    num_adults = q.passengers.get('adults', 1)
    num_children = q.passengers.get('children', 0)
    num_infants = q.passengers.get('infants', 0)

    if (
            (num_adults > 0 and num_children > 0)
            or (num_adults > 0 and num_infants > 0)
            or (num_children > 0 and num_infants > 0)
    ):
        raise QueryIsNotValid(
            'You can only choose one of "Traveler Type"(Adult/Child/Infant) in a request.')


def get_data(tracker, q):
    config = CONFIGS_BY_NV[q.national_version]

    r = tracker.wrap_request(
        requests.post,
        SERVICE_URL,
        params={
            'format': 'json',
            'ICODE': '758b8294b9074334aeecf86f531661d3',
            'Token': _access_token(tracker, config),
            'AID': config.aid,
            'SID': config.sid,
            'UUID': uuid4(),
        },
        json=request_json(q),
    )
    data = ujson.loads(r.content)
    return data


def parse_response(data, q):
    variants = []
    if 'ErrCode' in data or 'ErrMsg' in data:
        if data.get('ErrMsg') == 'INVALID_TOKEN':
            log.exception('Invalid authentication token')
        else:
            log.error('Api response Error: %s', data.get('ErrMsg'))
        return variants
    if not data['ResultCount']:
        return variants

    meta_url = data['BookingParametersInfo']['MetaURL']

    forward_products_by_group_id = {}
    backward_products_by_group_id = {}

    for i, product_info in enumerate(data['FltProductInfoList']):
        raw_group_id = product_info.get('GroupID')
        if raw_group_id is None:
            group_ids = [i]
        else:
            group_ids = raw_group_id.split(',')
        for group_id in group_ids:
            forward_products_by_group_id[group_id] = product_info

    for product_info in data.get('ReturnFltProductInfoList', []):
        raw_group_id = product_info['GroupID']
        group_ids = raw_group_id.split(',')
        for group_id in group_ids:
            backward_products_by_group_id[group_id] = product_info

    for group_id, forward_product in sleep_every(forward_products_by_group_id.iteritems()):
        backward_product = backward_products_by_group_id.get(group_id)

        if q.date_backward and backward_product is None:
            log.warning(
                'Backward product for group %s is expected but was not found. Skipping this group.',
                group_id,
            )
            continue

        forward_policy_info = forward_product['PolicyInfo'][0]
        backward_policy_info = backward_product['PolicyInfo'][0] if backward_product else None

        forward_baggage_info_list = forward_product.get('BaggageInfoList', [])
        backward_baggage_info_list = (
            backward_product.get('BaggageInfoList', []) if backward_product else None
        )

        v = Variant()

        # если есть обратный рейс, то берём policy_info из него.
        policy_info = backward_policy_info if backward_policy_info else forward_policy_info

        baggage_info_list = (
            backward_baggage_info_list if backward_baggage_info_list else forward_baggage_info_list
        )

        price_info = policy_info['PriceInfoList'][0]
        v.tariff = Price(
            float(price_info['TotalPrice']) * sum(
                int(cnt) for cnt in q.passengers.values() if str(cnt).isdigit()),
            price_info['Currency'],
        )

        if baggage_info_list is not None and len(baggage_info_list) > 0:
            forward_baggage_info = baggage_info_list[0]
        else:
            forward_baggage_info = None

        if baggage_info_list is not None and len(baggage_info_list) > 1:
            backward_baggage_info = baggage_info_list[1]
        else:
            backward_baggage_info = None

        v.forward.segments = list(_iterate_segments_parsed_from_product(
            forward_product, forward_baggage_info, q.importer.flight_fabric,
        ))

        if q.date_backward and backward_product:
            v.backward.segments = list(_iterate_segments_parsed_from_product(
                backward_product, backward_baggage_info, q.importer.flight_fabric,
            ))

        v.klass = q.klass  # Todo
        v.url = deeplink(
            meta_url, policy_info['ShoppingID'], v.tariff.value, q.national_version
        )
        v.order_data = {'url': v.url}
        variants.append(v)

    return variants


def deeplink(meta_url, shopping_id, amt, nv):
    if nv == 'ru':
        meta_url = meta_url.replace('www.trip.com', 'ru.trip.com', 1)
    if not meta_url.startswith('http'):
        meta_url = 'https://' + meta_url

    params = {
        'ShoppingID': shopping_id,
        'amt': amt,
    }
    return url_complement_missing(meta_url, params)


def parse_datetime(dt):
    """Parse strings with format: '/Date(1539673200000-0000)/'"""
    try:
        return datetime.fromtimestamp(int(dt[6:-10])) + timedelta(
            hours=5)  # получаем локальное время, смещённое на 5 часов
    except Exception:
        log.error('Cant parse datetime %r', dt)


def _parse_baggage_info(baggage_info):
    if not baggage_info or 'CheckedFormatted' not in baggage_info:
        return Baggage.from_partner()

    try:
        pieces = baggage_info['CheckedFormatted']['AdultDetail']['Piece']
        weight = baggage_info['CheckedFormatted']['AdultDetail']['Weight']

        if weight <= 0:
            return Baggage.from_partner()

        if pieces > 0:
            return Baggage.from_partner(pieces, weight)
        if pieces < 0:
            return Baggage.from_partner(weight=weight)

        return Baggage.from_partner(0)

    except Exception:
        log.exception('Cannot parse baggage')


def _iterate_segments_parsed_from_product(product, baggage_info, flight_fabric):
    for column_info in product['ColunmInfoList']:
        yield flight_fabric.create(
            company_iata=column_info['AirLine']['Code'],
            pure_number=int(column_info['FlightNo'][2:]),
            station_from_iata=column_info['DPort']['Code'],
            station_to_iata=column_info['APort']['Code'],
            local_departure=parse_datetime(column_info['DDate']),
            local_arrival=parse_datetime(column_info['ADate']),
            klass=column_info['Class'].lower(),
            baggage=_parse_baggage_info(baggage_info),
        )


def _get_site_version(national_version):
    if national_version == 'com':
        return 'EnglishSite'

    if national_version == 'tr':
        return 'TRSite'

    return 'RUSite'
