# -*- coding: utf-8 -*-
import re
import urllib2
from datetime import date, datetime
from itertools import product
from logging import getLogger
from urllib import urlencode

import requests
from lxml import etree

from travel.avia.library.python.common.utils.http import urlopen
from travel.avia.ticket_daemon.ticket_daemon.api.flights import IATAFlight, Variant
from travel.avia.ticket_daemon.ticket_daemon.daemon.utils import sleep_every
from travel.avia.ticket_daemon.ticket_daemon.lib.currency import Price
from travel.avia.ticket_daemon.ticket_daemon.lib.tracker import QueryTracker

log = getLogger(__name__)

DAVS_URL = 'http://xml.davs.ru/amadeus/xml/index.php'
DAVS2_URL = 'http://xml.davs.ru/gate/index.php'
DAVS_ORDER_TIMEOUT = 15 * 60
DAVS2_REDIRECT_URL = 'http://ticket.davs.ru/avia/book.php'


@QueryTracker.init_query
def query(tracker, q):
    # если запрос на сегодня, то спрашивать не надо (RASP-7031)
    if q.date_forward == date.today():
        return []

    params = {
        'ip': '87.250.250.34',  # rasp.yandex.ru
        'DepartureDate': q.date_forward.strftime('%d/%m/%Y'),
        'From': q.iata_from,
        'To': q.iata_to,
        'ADT': q.passengers.get('adults', 0),
        'CHD': q.passengers.get('children', 0),
        'INF': q.passengers.get('infants', 0),
        'oneway': '1' if not q.date_backward else '0',
        'ItineraryCabinOption': {
            'economy': 'Y',
            'business': 'C',
            'first': 'F',
        }[q.klass],
        'date_type': 'exactly'
    }

    if q.date_backward:
        params['ArrivalDate'] = q.date_backward.strftime('%d/%m/%Y')

    davs = DavsRequest(action='dep', params=params)

    xml = davs.get_data(tracker, q)

    variants = parse_response(xml, q, params)

    return variants


def parse_response(response, q, params):
    tree = response.tree

    if response.error:
        return []

    variants = []

    for variant_item in sleep_every(tree.xpath('//items/item[@price]')):
        try:
            variants += parse_variants(variant_item, q, params) or []

        except Exception:
            log.warning(u"Ошибка разбора варианта")

    return variants


def parse_variants(item_element, q, params):
    variants = []

    raw_data = etree.tostring(
        item_element, encoding=unicode, pretty_print=True)

    tariff = float(item_element.get('price'))

    bounds = item_element.findall('./bound')

    forward_elements = parse_elements(bounds[0])
    backward_elements = {}

    if len(bounds) >= 2:
        backward_elements = parse_elements(bounds[1])

    if forward_elements and backward_elements:
        combinations_tags = item_element.findall('./combinations/combination')

        if combinations_tags:
            icombinations = (c.text.split(',') for c in combinations_tags)

        else:
            icombinations = product(
                forward_elements.keys(),
                backward_elements.keys()
            )

        for forward_id, backward_id in icombinations:
            v = Variant()
            v.raw_data = raw_data
            v.tariff = Price(tariff)
            v.klass = q.klass
            v.params = params

            v.forward.segments = forward_elements[forward_id]
            v.backward.segments = backward_elements[backward_id]

            v.order_data = {
                'elements': '%s|%s' % (forward_id, backward_id),
            }

            v.order_data.update(params)

            variants.append(v)

    else:
        for element_id, segments in forward_elements.items():
            v = Variant()
            v.raw_data = raw_data
            v.tariff = Price(tariff)
            v.klass = q.klass
            v.params = params

            v.forward.segments = segments

            v.order_data = {
                'elements': element_id,
            }
            v.order_data.update(params)

            variants.append(v)

    return variants


ROUTE_CODE_RE = re.compile(ur'([a-zа-я0-9]{2,3}?)(\d+)', re.I | re.U)


def parse_elements(bound_elem):
    elements = {}

    for element in bound_elem.findall('./element'):
        flights = []

        for segment in element.findall('./segment'):
            f = IATAFlight()

            f.station_from_iata = segment.find('./departure/airport').get('id')
            f.station_to_iata = segment.find('./arrival/airport').get('id')

            company_iata = segment.find('./supplier').get('id')

            flight_number = segment.find('./carrier_number').text

            f.number = "%s %s" % (company_iata, flight_number)

            f.company_iata = company_iata

            f.local_departure = datetime.strptime(segment.find(
                './departure/date').text + 'T' + segment.find('./departure/time').text, '%Y-%m-%dT%H:%M')
            f.local_arrival = datetime.strptime(segment.find(
                './arrival/date').text + 'T' + segment.find('./arrival/time').text, '%Y-%m-%dT%H:%M')

            f.electronic_ticket = False

            flights.append(f)

        elements[element.get('id')] = flights

    return elements


def find_station(station_title):
    if re.search('<!--A:([A-Z]+)-->', station_title):
        iata = re.search('<!--A:([A-Z]+)-->', station_title).group(1)

        return {'iata': iata}

    elif re.search(ur'([а-я -]+) \(([а-я -]+)\)', station_title, re.I | re.U):
        m = re.search(ur'([а-я -]+) \(([а-я -]+)\)',
                      station_title, re.I | re.U)
        city = m.group(1)
        station = m.group(2)

        return {'city_title': (city, station)}

    return {}


class DavsError(Exception):
    pass


class DavsRequest:
    def __init__(
        self, xml_gate_url=DAVS2_URL,
        action=None, session=None, params=None
    ):
        self.url = xml_gate_url

        # параметры запроса
        self.params = params or {}

        if not action:
            raise ValueError('Укажите action')

        self.params['action'] = action
        self.params['site'] = 'yandex.ru'

        if session:
            self.params['session'] = session

    def set_param(self, key, value):
        self.params[key] = value

    def send(self):
        request = urllib2.Request(
            self.url, self.build_query()
        )

        response = urlopen(request, timeout=DAVS_ORDER_TIMEOUT)

        return DavsResponse(response)

    def get_data(self, tracker, q):
        r = tracker.wrap_request(
            requests.post,
            self.url,
            headers={
                'Content-Type': 'application/x-www-form-urlencoded; charset=Windows-1251',
            },
            data=self.build_query(),
        )

        return DavsResponse(r.text.encode('cp1251'))

    def build_query(self):
        body = '<request>'

        for k, v in self.params.items():
            body += '<%(tag)s>%(value)s</%(tag)s>' % {'tag': k, 'value': v}

        body += '</request>'

        return urlencode({'req_xml': body.encode('cp1251')})


class DavsResponse:
    # правильная последовательность действий
    actions_sequence = ['dep', 'book', 'name']

    def __init__(self, response):
        self.tree = (
            etree.fromstring(response) if isinstance(response, basestring)
            else etree.parse(response)
        )

        try:
            self.action = self.tree.xpath('/data/action')[0].text
            self.session = self.tree.xpath('/data/session')[0].text
            self.response = self.tree.xpath('/data/response')[0]
            self.next_action = self.tree.xpath(
                '/data/response/next_action')[0].text

            current_action_idx = self.actions_sequence.index(self.action)
            next_action_idx = self.actions_sequence.index(self.next_action)

            self.error = False

            if next_action_idx != current_action_idx + 1:
                self.error = True

        except IndexError:
            self.error = True


class DavsBooking:

    def __init__(self, xml_gate_url=DAVS_URL,
                 session=None,
                 davs_route_id=None,
                 **kwargs):
        if not session:
            raise ValueError('укажите id сессии davs')

        if not davs_route_id:
            raise ValueError('укажите id рейса для бронирования')

        self.request = DavsRequest(
            xml_gate_url=xml_gate_url,
            action='book',
            session=session
        )
        self.request.params['reis'] = davs_route_id

        for k, v in kwargs.items():
            self.request.params[k] = v

    def book(self):
        return self.request.send()


def book(order_data):
    elements = order_data['elements'].split('|')
    del order_data['elements']  # Это QueryDict так что pop не работает :(

    order_data['ip'] = '127.0.0.1'

    request = DavsRequest(action='dep', params=order_data)
    response = request.send()
    session = response.session

    params = {
        'PHPSESSID': session,
    }

    for index, element in enumerate(elements):
        params['element[%s]' % index] = element

    url = DAVS2_REDIRECT_URL + '?' + urlencode(params)

    return {
        'url': url,
        'm_url': url + '&mobile=1',
    }
