# -*- coding: utf-8 -*-
from datetime import datetime
from logging import getLogger

import requests
from lxml import etree

from travel.avia.ticket_daemon.ticket_daemon.api.flights import Variant
from travel.avia.ticket_daemon.ticket_daemon.daemon.utils import sleep_every
from travel.avia.ticket_daemon.ticket_daemon.lib.currency import Price
from travel.avia.ticket_daemon.ticket_daemon.lib.tracker import QueryTracker
from travel.avia.ticket_daemon.ticket_daemon.lib.baggage import Baggage

log = getLogger(__name__)

DELTATOURS_URL = 'http://online2.delta-tours.ru/yandex_flight'


@QueryTracker.init_query
def query(tracker, q):
    xml = get_data(tracker, q)
    variants = list(parse_response(xml, q))

    return variants


def get_data(tracker, q):
    params = {
        'departure': q.iata_from,
        'arrival': q.iata_to,
        'date_forward': q.date_forward.strftime('%Y-%m-%d'),
        'class': {'economy': 'E', 'business': 'B'}[q.klass],
        'adult': q.passengers.get('adults', 0),
        'child': q.passengers.get('children', 0),
        'infant': q.passengers.get('infants', 0),
        'REFID': 'Yandex',
    }

    if q.date_backward:
        params['date_backward'] = q.date_backward.strftime('%Y-%m-%d')

    r = tracker.wrap_request(
        requests.get,
        DELTATOURS_URL,
        params=params
    )

    return r.content


# (E - эконом, B - бизнес, F – первый класс, P – премиум)
SERVCE_CLASS_MAP = {
    'Y': 'economy',
    'E': 'economy',
    'B': 'business',
    'F': 'first',
    'P': 'first',  # Премиум тоже считаем за первый
}


def parse_response(xml, q):
    tree = etree.fromstring(xml)

    for v_tag in sleep_every(tree.xpath('//variants/variant')):
        v = Variant()

        v.forward.segments = list(parse_flights(
            v_tag.xpath('route_forward'), q.importer.flight_fabric))

        if q.date_backward:
            v.backward.segments = list(parse_flights(
                v_tag.xpath('route_backward'), q.importer.flight_fabric))

        fare_tag = v_tag.find('fare')

        v.klass = SERVCE_CLASS_MAP[fare_tag.get('class')]

        price_val = float(fare_tag.get('value').replace(',', '.'))
        v.tariff = Price(price_val, 'RUR')

        v.url = v_tag.get('url')
        v.order_data = {'url': v.url}

        yield v


def parse_flights(segment_tags, flight_fabric):
    for f_tag in segment_tags:
        company_code = f_tag.get('company_code')
        yield flight_fabric.create(
            station_from_iata=f_tag.get('departure_airport_code'),
            station_to_iata=f_tag.get('arrival_airport_code'),
            local_departure=datetime.strptime(f_tag.get('departure_datetime'),
                                              '%Y-%m-%d %H:%M:%S'),

            local_arrival=datetime.strptime(f_tag.get('arrival_datetime'),
                                            '%Y-%m-%d %H:%M:%S'),
            company_iata=company_code,
            pure_number=f_tag.get('route_code').lstrip(company_code),
            baggage=Baggage.from_partner(weight=int(f_tag.get('baggage'))),
        )
