# -*- coding: utf-8 -*-
import random
from collections import namedtuple
from datetime import date, datetime
from logging import getLogger

import requests

from travel.avia.library.python.common.models.partner import DohopVendor, Partner

from travel.avia.ticket_daemon.ticket_daemon.api.flights import Variant
from travel.avia.ticket_daemon.ticket_daemon.api.query import QueryIsNotValid
from travel.avia.ticket_daemon.ticket_daemon.api.redirect import RedirectError
from travel.avia.ticket_daemon.ticket_daemon.daemon.utils import (
    BadPartnerResponse, async_sleep, eat_exception
)
from travel.avia.ticket_daemon.ticket_daemon.lib.currency import Price
from travel.avia.ticket_daemon.ticket_daemon.lib.decorators import elementwise
from travel.avia.ticket_daemon.ticket_daemon.lib.partner_secret_storage import partner_secret_storage
from travel.avia.ticket_daemon.ticket_daemon.lib.tracker import QueryTracker
from travel.avia.ticket_daemon.ticket_daemon.partners import PartnerBookNotFoundException

log = getLogger(__name__)

API_ROOT = 'https://yandexapi.dohop.com'
API_URL = '%s/api/v1' % API_ROOT
POOL_ROOT = '%s/poll' % API_URL
LINK_ROOT = '%s/transfer' % API_URL
MAX_REQ_TIME = 60


RESIDENCIES_MAP = {'ru': 'RU', 'ua': 'UA', 'tr': 'TR', 'com': 'GB'}
CURRENCIES_MAP = {'ru': 'RUR', 'ua': 'UAH', 'tr': 'TRY', 'com': 'EUR'}

VENDOR_ID_BY_PARTNER_CODE = {
    'ottselfbook': 1262,
}
PARTNER_CODE_BY_VENDOR_ID = {v: k for k, v in VENDOR_ID_BY_PARTNER_CODE.items()}

SEARCH_PARTNER = partner_secret_storage.get(
    importer_name='dohop', namespace='PASSWORD',
)


class ParsingFareException(Exception):
    pass


def validate_query(q):
    if q.klass != 'economy':
        raise QueryIsNotValid('Only economy requests allowed')

    if not q.station_iatas_from:
        raise QueryIsNotValid('No station_iatas_from: %r' % q.iata_from)

    if not q.station_iatas_to:
        raise QueryIsNotValid('No station_iatas_to: %r' % q.iata_to)

    q.partner_codes = set([
        p.code for p in q.importer.partners
        if isinstance(p, Partner) and p.code in VENDOR_ID_BY_PARTNER_CODE
    ])
    q.include_vendors = [
        int(v.dohop_id) for v in q.importer.partners
        if isinstance(v, DohopVendor)
    ] + [
        VENDOR_ID_BY_PARTNER_CODE[code] for code in q.partner_codes
    ]

    q.dohop_vendors_cache_ttl = {
        int(v.dohop_id): v.dohop_cache_ttl
        for v in q.importer.partners if isinstance(v, DohopVendor)
    }


def not_adults_ages(q):
    children_ages = ['11'] * q.passengers.get('children', 0)
    infants_ages = ['1'] * q.passengers.get('infants', 0)

    return children_ages + infants_ages


def build_passengers_params(q):
    passengers_params = {}
    youngster_ages = not_adults_ages(q)
    passengers_params['n_adults'] = q.passengers.get('adults', 0)

    if youngster_ages:
        passengers_params['youngsters_ages'] = ','.join(youngster_ages)

    return passengers_params


def _index_by_national_version(national_version):
    return 'yandexcom' if national_version == 'com' else 'yandex-multiticket'


def _get_residence_code(q):
    return RESIDENCIES_MAP.get(q.national_version, RESIDENCIES_MAP['ru'])


def init_search(tracker, q):
    residence_code = _get_residence_code(q)
    log.debug('Residence code: %r %r', residence_code, random.random())

    init_url = '{}/{}/{}/{}/{}/{}/{}{}'.format(
        'https://yandexapi.dohop.com/api/v2/search',
        _index_by_national_version(q.national_version),
        q.lang,
        residence_code,
        ','.join(q.station_iatas_from),
        ','.join(q.station_iatas_to),
        q.date_forward.strftime('%Y-%m-%d'),
        '/{}'.format(q.date_backward.strftime('%Y-%m-%d')) if q.date_backward else '',
    )
    log.debug('init_url: %s', init_url)

    init_params = build_passengers_params(q)
    init_params['faredata_cp_group_limit'] = 0
    init_params['search-partner'] = SEARCH_PARTNER

    if q.include_vendors:
        str_ids = ','.join(map(str, q.include_vendors))
        init_params['include_vendors'] = str_ids
        init_params['allow_unverified_fares_from_vendors'] = str_ids

    try:
        r = tracker.wrap_request(requests.get, init_url, params=init_params)

    except BadPartnerResponse as e:
        if e.response.status_code == 400:
            init_json = e.response.json()
            error_text = init_json.get('error')

            log.warning(
                '%s init error: %s %s %r',
                q.qid_msg, init_url, init_json.get('error', 'unknown'), error_text
            )

        raise

    log.debug('%s init: %s', q.qid_msg, init_url)

    init_json = r.json()

    key = init_json['key']
    pooler = pool(tracker, q, key, init_json['continuation'])

    return (key, pooler)


def pool(tracker, q, key, continuation):
    while True:
        pool_url = '%s/%s/%s' % (POOL_ROOT, key, continuation)
        log.debug('%s pool: %s', q.qid_msg, pool_url)
        r = tracker.wrap_request(requests.get, pool_url)
        r.raise_for_status()

        data_json = r.json()
        fatal_error = data_json.get('fatal_error')
        if fatal_error:
            log.warning(fatal_error)
            raise BadPartnerResponse('dohop', r)

        continuation = data_json.get('continuation')

        yield data_json

        if data_json.get('is_done'):
            break


@QueryTracker.init_query
def query(tracker, q):
    key, pooler = init_search(tracker, q)
    return gen_variants_chunks(tracker, q, key, pooler)


def gen_variants_chunks(tracker, q, key, pooler):
    processed_fares = set()

    r_flights_by_key = {}

    r_fares = []

    unverified_coroutine = run_unverified_coroutine(q, key)
    next(unverified_coroutine)  # returns unused first yield result (==None)

    flight_fabric = q.importer.flight_fabric

    for data_json in pooler:
        if 'search' in data_json:
            raw_flights = data_json['search']['flights']
            flights = [
                parse_dohop_flight(flight_fabric, f) for f in raw_flights
            ]

            d_flights = {_dohop_flight_key(df.legs[0]): df for df in flights}
            # assert not (set(d_flights) & set(r_flights_by_key)), 'Dohop flight parsed two times'
            for df_key, df in d_flights.iteritems():
                if df_key in r_flights_by_key:
                    log.warning('Dohop flight parsed two times')
            r_flights_by_key.update(d_flights)

        variants = []

        if 'fares' in data_json:
            all_new_fares = filter(None, (
                parse_fare(q, r_flights_by_key, int(k), v)
                for k, v in data_json['fares'].iteritems()
            ))

            r_fares.extend(f for f in all_new_fares if f.vendor_fares_nonexpired)

        variants.extend([
            variant
            for fare in r_fares
            for variant in make_variants(
                fare, key, q, processed_fares
            )
        ])

        variants.extend(unverified_coroutine.send(data_json))

        if variants:
            yield variants


def run_unverified_coroutine(q, key):
    processed_unverified_fares = set()
    result = None

    while True:
        data_json = yield result

        unverified = data_json.get('unverified')
        if not unverified:
            result = []
            continue

        unverified_variants = filter(None, parse_unverified_variants(
            unverified, q, key, processed_unverified_fares
        ))
        log.info('Got unverified variants: %r %r', q.id, [
            (
                v.partner_code,
                (v.tariff.value, v.tariff.currency),
                v.klass,
                tuple(
                    (
                        f.number,
                        f.local_departure and f.local_departure.strftime('%Y-%m-%d %H:%M'),
                        f.local_arrival and f.local_arrival.strftime('%Y-%m-%d %H:%M'),
                        f.station_from_iata,
                        f.station_to_iata,
                    )
                    for f in v.iter_all_segments()
                ),
            )
            for v in unverified_variants
        ])
        result = unverified_variants


@elementwise
def parse_unverified_variants(d, q, key, processed_unverified_fares):
    """
    {
        "transfer-id": "_0",
        "fare": {
            "v": 1262,
            "a": 68740,
            "c": "ISK",
            "d": "2016-11-17 16:29:38",
            "f": 67067,
        },
        "flights_out": [
            ["CDG", "LHR", "AF1844", "2016-12-18 16:00", "2016-12-18 21:40", 220],
            ["LHR", "SVO", "AF1845", "2016-12-18 22:00", "2016-12-19 12:40", 320],
        ],
        "flights_home": [
            ["SVO", "CDG", "AF1846", "2016-12-22 16:00", "2016-12-18 21:40", 220],
        ],
    },
    """
    async_sleep(0)

    vendor_id = int(d['fare']['v'])
    transfer_id = d['transfer-id']
    vendor_fare_key = (transfer_id, vendor_id)
    if vendor_fare_key in processed_unverified_fares:
        return None

    try:
        # fare_age дохопа — в секундах
        allowed_vendor_fare_age = 60 * q.dohop_vendors_cache_ttl[vendor_id]
    except KeyError:
        return None

    fare_age = int(d['fare']['a'])
    if fare_age > allowed_vendor_fare_age:
        log.debug(
            '%s Skip vendor %r fare cause of fare age: %s > %s',
            q.qid_msg, vendor_id, fare_age, allowed_vendor_fare_age
        )
        return None

    v = Variant()

    flight_fabric = q.importer.flight_fabric
    v.forward.segments = parse_unv_segments(d['flights_out'], flight_fabric)

    if d.get('flights_home'):
        v.backward.segments = parse_unv_segments(d['flights_home'], flight_fabric)

    v.tariff = Price(float(d['fare']['f']), d['fare']['c'])
    v.klass = q.klass

    if PARTNER_CODE_BY_VENDOR_ID.get(vendor_id) in q.partner_codes:
        v.partner_code = PARTNER_CODE_BY_VENDOR_ID[vendor_id]
    else:
        v.dohop_vendor_id = vendor_id
        v.partner_code = 'dohop_%s' % v.dohop_vendor_id

    v.order_data = {
        'end_point_url': API_URL +
        '/transfer/{key}/{transfer_id}/{vendor_id}/{currency}'.format(
            key=key,
            transfer_id=transfer_id,
            vendor_id=d['fare']['v'],
            currency=d['fare']['c'],
        ),
    }

    v.dohop_unverified = True

    processed_unverified_fares.add(vendor_fare_key)

    return v


@elementwise
def parse_unv_segments(raw_segment, flight_fabric):
    """
    Each: ["SVO", "CDG", "AF1846", "2016-12-22 16:00", "2016-12-18 21:40", 220]
    """
    dep_iata, arr_iata, fnum, departue_dt, arrival_dt, path_time = raw_segment
    company_iata = fnum[:2]
    flight_number = '%s %s' % (company_iata, fnum[2:])

    return flight_fabric.create(
        company_iata=company_iata,
        number=flight_number,

        station_from_iata=dep_iata,
        station_to_iata=arr_iata,

        local_departure=datetime.strptime(departue_dt, '%Y-%m-%d %H:%M'),
        local_arrival=datetime.strptime(arrival_dt, '%Y-%m-%d %H:%M'),
    )


def make_variants(fare, key, q, processed_fares):
    filtered_fares = []

    for vfare in fare.vendor_fares_nonexpired:
        # TODO maybe move upper
        vendor_fare_key = (fare.id, vfare.vendor_id)
        if vendor_fare_key in processed_fares:
            continue

        processed_fares.add(vendor_fare_key)
        filtered_fares.append(vfare)

    if not filtered_fares:
        return

    forward_segments = [f.iata_flight for f in fare.outbound_flights]
    backward_segments = [f.iata_flight for f in fare.homebound_flights]

    # for price, vendor_currency, normalized_currency, vendor_id in filtered_fares:
    for vfare in filtered_fares:
        async_sleep(0)
        v = Variant()

        v.forward.segments = forward_segments
        v.backward.segments = backward_segments

        v.tariff = Price(vfare.tariff.value, vfare.tariff.currency)

        v.klass = q.klass

        if PARTNER_CODE_BY_VENDOR_ID.get(vfare.vendor_id) in q.partner_codes:
            v.partner_code = PARTNER_CODE_BY_VENDOR_ID[vfare.vendor_id]
        else:
            v.dohop_vendor_id = vfare.vendor_id
            v.partner_code = 'dohop_%s' % v.dohop_vendor_id

        v.order_data = {

            # Удалить когда подхватятся новые параметры
            'end_point_url': '{}/{}/{}/{}/{}'.format(
                LINK_ROOT, key, fare.id, vfare.vendor_id,
                rur2rub(vfare.tariff.currency)
            ),

            'key': key,
            'fareId': fare.id,
            'vendorId': vfare.vendor_id,
        }

        yield v


def rur2rub(currency):
    return {'RUR': 'RUB'}.get(currency, currency)


def rub2rur(currency):
    return {'RUB': 'RUR'}.get(currency, currency)


Fare = namedtuple('Fare', [
    'id', 'outbound_flights', 'homebound_flights',
    'vendor_fares', 'vendor_fares_nonexpired'
])


@eat_exception
def parse_fare(q, r_flights_by_key, fid, raw):
    outbound_flights = parse_fare_flights(r_flights_by_key, raw[u'o'])
    homebound_flights = parse_fare_flights(r_flights_by_key, raw[u'h'])
    vfares = tuple(
        parse_vendor_fare(q, int(vendor_id), vfare_raw)
        for vendor_id, vfare_raw in raw[u'f'].iteritems()
        if int(vendor_id) in q.include_vendors
    )

    return Fare(
        fid,
        outbound_flights=outbound_flights,
        homebound_flights=homebound_flights,
        vendor_fares=vfares,
        vendor_fares_nonexpired=tuple(vf for vf in vfares if not vf.is_expired),
    )


def is_vendor_fare_expired(q, vendor_id, fare_age):
    try:
        # fare_age дохопа — в секундах
        allowed_vendor_fare_age = 60 * q.dohop_vendors_cache_ttl[vendor_id]
    except KeyError:
        return False

    if fare_age > allowed_vendor_fare_age:
        log.debug(
            '%s Vendor %r fare expired cause of age %s > %s',
            q.qid_msg, vendor_id, fare_age, allowed_vendor_fare_age
        )
        return True

    return False


VendorTariff = namedtuple('VendorTariff', ['value', 'currency'])


VendorFare = namedtuple('VendorFare', [
    'vendor_id', 'tariff', 'dep_dt', 'age', 'is_expired'
])


def parse_vendor_fare(q, vendor_id, raw):
    age = int(raw[u'a'])
    return VendorFare(
        vendor_id,
        tariff=VendorTariff(value=float(raw[u'f']), currency=rub2rur(raw[u'c'])),
        dep_dt=datetime.strptime(raw['d'], '%Y-%m-%d %H:%M:%S'),
        age=age,
        is_expired=is_vendor_fare_expired(q, vendor_id, age)
    )


def parse_fare_flights(r_flights_by_key, plain_tetrads):
    return tuple(
        parse_fare_flight(r_flights_by_key, raw_fare_flight)
        for raw_fare_flight in ipaginate(plain_tetrads, 4)
    )


def ipaginate(indexable, pagesize):
    return (indexable[p:p+pagesize]
            for p in xrange(0, len(indexable), pagesize))


def _dohop_flight_key(f):
    return (f.company_iata, f.flight_number, f.dep_day)


def parse_fare_flight(r_flights_by_key, raw_fare_flight):
    assert len(raw_fare_flight) == 4
    dep_iata, arr_iata, flight_number, raw_dep_day = raw_fare_flight
    dep_day = date(*map(int, raw_dep_day.split('-')))
    company_iata, flight_number = parse_flight_number(flight_number)

    # Should be like def _dohop_flight_key(f):
    dohop_flight_key = (company_iata, flight_number, dep_day)
    if dohop_flight_key not in r_flights_by_key:
        log.warning('Parsing fare for unknown flight: %r', dohop_flight_key)
        raise ParsingFareException('Unknown flight: %r' % dohop_flight_key)

    return r_flights_by_key[dohop_flight_key]

    # return FareFlight(
    #     flight=r_flights_by_key[dohop_flight_key],
    #     dep_iata=dep_iata,
    #     arr_iata=arr_iata,
    #     company_iata=company_iata,
    #     flight_number=flight_number,
    #     dep_day=dep_day
    # )


# FareFlight = namedtuple('FareFlight', [
#     'flight', 'dep_iata', 'arr_iata', 'company_iata', 'flight_number', 'dep_day'
# ])


def parse_flight_number(text):
    company_iata = text[:2]
    flight_number = '{} {}'.format(company_iata, text[2:])
    return company_iata, flight_number


DohopLeg = namedtuple('DohopLeg', [
    'dep_iata', 'arr_iata',
    'company_iata', 'flight_number',
    'dep_day', 'dep_dt', 'arr_dt',
])


def parse_dohop_leg(raw_leg):
    assert len(raw_leg) == 7
    (dep_iata, arr_iata, flight_number, dep_dt, arr_dt,
     _duration, _aircraft,) = raw_leg

    company_iata, flight_number = parse_flight_number(flight_number)

    dep_dt = datetime.strptime(dep_dt, '%Y-%m-%d %H:%M')

    return DohopLeg(
        dep_iata=dep_iata,
        arr_iata=arr_iata,
        company_iata=company_iata,
        flight_number=flight_number,
        dep_day=dep_dt.date(),
        dep_dt=dep_dt,
        arr_dt=datetime.strptime(arr_dt, '%Y-%m-%d %H:%M'),
    )


DohopFlight = namedtuple('DohopFlight', ('legs', 'iata_flight'))


def parse_dohop_flight(flight_fabric, raw_dohop_flight):
    assert len(raw_dohop_flight) == 5
    _published, _operating, legs, _duration, _ttl = raw_dohop_flight
    legs = tuple(parse_dohop_leg(s) for s in legs)
    first_leg = legs[0]
    final_leg = legs[-1]

    return DohopFlight(
        legs=legs,
        iata_flight=flight_fabric.create(

            company_iata=first_leg.company_iata,
            number=first_leg.flight_number,

            station_from_iata=first_leg.dep_iata,
            station_to_iata=final_leg.arr_iata,

            local_departure=first_leg.dep_dt,
            local_arrival=final_leg.arr_dt,

        )
    )


def poll_deeplink(deeplink_uri_poll):
    poll_url = '%s%s' % (API_ROOT, deeplink_uri_poll)
    log.info('get deeplink-uri-poll: %s', poll_url)

    try:
        r = requests.get(poll_url)
    except Exception as e:
        log.warning('Dohop poll deeplink error: %r %r', poll_url, e)
        raise
    if not r.ok:
        log.warning('Dohop poll deeplink error: %r %r %r', poll_url, r.status_code, r.content)
        r.raise_for_status()
    polled = r.json()
    log.info('deeplink-uri-poll received: %r', polled)

    return polled['deeplink-url']


def book(order_data):
    qid_msg = order_data.get('qkey')

    key = order_data.get('key')
    fare_id = order_data.get('fareId')
    vendor_id = order_data.get('vendorId')
    currency = order_data.get('currency')
    currency = {'RUR': 'RUB'}.get(currency, currency)
    if key and fare_id and vendor_id and currency:
        endpoint_url = '{}/{}/{}/{}/{}'.format(
            LINK_ROOT, key, fare_id, vendor_id, currency)

    else:
        endpoint_url = order_data['end_point_url']

    log.info('%s get end_point_url: %s', qid_msg, endpoint_url)

    try:
        r = requests.get(endpoint_url)
    except Exception as e:
        log.warning('Dohop endpoint error: %r %r', endpoint_url, e)
        raise
    if not r.ok:
        log.warning('Dohop endpoint error: %r %r %r', r.status_code, endpoint_url, r.content)

        # Если данные для редиректа не нашлись, то мы ничего не можем сделать,
        # поэтому не стоит падать
        if r.status_code == 400:
            raise PartnerBookNotFoundException(
                order_data['partner'],
                qid=qid_msg
            )

        r.raise_for_status()
    endpoint_answer = r.json()

    log.debug('Endpoint answer: %r', endpoint_answer)

    def _get_deeplink_data():
        # Есть прямой URL
        if 'deeplink-url' in endpoint_answer:
            deeplink_url = endpoint_answer['deeplink-url']
            log.debug('%s use existed: %r', qid_msg, deeplink_url)
            return deeplink_url

        elif 'deeplink-uri-poll' in endpoint_answer:
            deeplink_uri_poll = endpoint_answer['deeplink-uri-poll']
            try:
                deeplink_data = poll_deeplink(deeplink_uri_poll)
                log.info('%s use polled: %r', qid_msg, deeplink_data)
                return deeplink_data
            except Exception as e:
                log.warning('Dohop poll deeplink fail: %r %r', deeplink_uri_poll, e)

        # Запасной вариант
        if 'deeplink-url-fallback' in endpoint_answer:
            deeplink_data = endpoint_answer['deeplink-url-fallback']
            log.info('%s fallback: %r', qid_msg, deeplink_data)
            return deeplink_data

    data = _get_deeplink_data()
    if not data:
        log.warning('%s No redirect data %r', qid_msg, order_data)
        raise RedirectError('No redirect data %s' % qid_msg)

    if 'post' in data:
        url, post = (data['url'], data['post'])
        log.info('redirect to: %r %r', url, post)
        return url, post
    else:
        url = data['url']
        log.info('redirect to: %r', url)
        return url

# debug


def repr_fl(f):
    return (
        f.legs[0].flight_number,
        f.legs[0].dep_iata,
        f.legs[-1].arr_iata,
    )


def repr_fare(f):
    return namedtuple('Fare', 'id hop out home')(
        f.id,
        fare_hops(f),
        [repr_fl(fl) for fl in f.outbound_flights],
        [repr_fl(fl) for fl in f.homebound_flights],
    )


def fare_hops(f):
    return ((f.outbound_flights[0].legs[0].dep_iata,
             f.outbound_flights[-1].legs[-1].arr_iata),
            (f.homebound_flights[0].legs[0].dep_iata,
             f.homebound_flights[-1].legs[-1].arr_iata) if f.homebound_flights else ()
            )
