"""
An importer for eTRAVELi clients.
Trip.ru (RASPTICKETS-15318)  and Supersaver (RASPTICKETS-15304) use it
"""

from datetime import datetime
from itertools import chain
import hashlib

import requests
from django.utils.http import urlencode
from typing import Dict

from travel.avia.ticket_daemon.ticket_daemon.api.flights import Variant
from travel.avia.ticket_daemon.ticket_daemon.api.query import QueryIsNotValid, get_service_by_qid
from travel.avia.ticket_daemon.ticket_daemon.daemon.utils import sleep_every
from travel.avia.ticket_daemon.ticket_daemon.lib.baggage import Baggage
from travel.avia.ticket_daemon.ticket_daemon.lib.currency import Price
from travel.avia.ticket_daemon.ticket_daemon.lib.http import url_complement_missing

KLASS_MAP = {
    'economy': 'Y',
    'business': 'P',
}

ONE_POUND_IN_KILO = 0.45

BRANDS_TO_COUNTRY_COMBINATIONS = {
    'airtickets24': 'FR',
    'avion': 'RO',
    'budjet': 'DK FI NO SE',
    'flygvaruhuset': 'SE',
    'goleif': 'DK',
    'gotogate': 'AE AT AU BE BG CA CH CN CZ DK ES FR GB GR HK HU ID IE IL IN IT JP KR MX MY NL NO NZ PH PL PT RO SA SG SK TH TR TW UA US VN ZA',
    # NOTE: Gotogate operates as flybillet in Denmark. For flybillet.dk content please use brand gotogate and country DK
    'mytrip': 'AR AT AU BE BO BR CA CH CL CN CO CU CZ DE DK EE ES FI FR GB GG HK HR HU ID IE IL IN IR IS IT JP KR LU MO MT MX MY NG NL NO NZ PE PH PL PT RS SE SG SK TH TR TW US UY VE VN ZA',
    'pamediakopes': 'CY GR',
    'seat24': 'DE DK FI NO SE',
    'supersaver': 'AE AT CA CH DK ES FI FR GB IE IT JP NL NO PL RU SE ZA',
    'travelfinder': 'SE',
    'travelpartner': 'FI NO SE',
    'travelstart': 'DE DK FI NO SE',
    'trip': 'AE BG BH BY EG JO KW KZ OM QA RU SA UA',
}  # type: Dict[str, str]

COM_COUNTRY = {
    'mytrip': 'GG',
}  # type: Dict[str, str]


def get_request_country(query, partner):
    if query.national_version == 'com':
        return COM_COUNTRY.get(partner, 'GB')
    else:
        return query.national_version.upper()


def _add_prefix(prefix, iterable):
    """ Add prefix to each element in iterable """
    for iter_ in iterable:
        yield prefix + str(iter_)


def _add_prefix_and_suffix(prefix, suffix, iterable):
    """ Add prefix and suffix to each element in iterable """
    for iter_ in iterable:
        yield prefix + str(iter_) + suffix


class ETravelIQueryBuilder(object):
    def __init__(self, base_url, partner):
        self._base_url = unicode(base_url)
        self._partner = partner

    def build(self, q):
        """
        Return url for ETrvelImporter
        :param q ticket_daemon.api.query instance
        :return str
        """
        return self._build_url(q)

    def _build_url(self, query):
        return self._base_url + u'?' + urlencode(self._build_params(query))

    @staticmethod
    def _serialize_date(date):
        return date.strftime('%Y-%m-%d')

    @staticmethod
    def _make_bound(iata_from, iata_to, date):
        return u'{iata_from}{iata_to}{date}'.format(
            iata_from=iata_from,
            iata_to=iata_to,
            date=ETravelIQueryBuilder._serialize_date(date),
        )

    @staticmethod
    def _make_bounds(q):
        bounds = [ETravelIQueryBuilder._make_bound(q.iata_from, q.iata_to, q.date_forward)]
        if q.date_backward:
            bounds.append(ETravelIQueryBuilder._make_bound(q.iata_to, q.iata_from, q.date_backward))

        return bounds

    @staticmethod
    def _make_travellers(adults, children, infants):
        return ','.join(chain(
            _add_prefix('a', range(1, adults + 1)),
            _add_prefix_and_suffix('c', ':10', range(1, children + 1)),  # Assume that all children are ten years old
            _add_prefix_and_suffix('i', ':2', range(1, infants + 1)),  # Assume that all infants are two years old
        ))

    def _build_params(self, q):
        """https://api.etraveli.com/ws/ultrasearch/2.0?brand=supersaver&country=fi&travellers=a1,a2,c1:8,c2:10&bounds=STOBKK2018-12-20,BKKSTO2019-01-05"""

        return {
            'brand': self._partner,
            'country': get_request_country(q, self._partner),
            'bounds': ','.join(self._make_bounds(q)),
            'travellers': self._make_travellers(q.adults, q.children, q.infants),
            'cabin': KLASS_MAP[q.klass],
            'searchId': hashlib.md5(q.id).hexdigest(),
            'format': 'json',
        }


class ETravelIImporter(object):
    BASE_URL = 'https://api.etraveli.com/ws/ultrasearch/2.0'

    def __init__(self, partner, login, password, logger):
        self._login = login
        self._password = password
        self._logger = logger
        self._brand = partner
        self._query_builder = ETravelIQueryBuilder(self.BASE_URL, partner)

    def query(self, tracker, q):
        response = self._get_data(tracker, q)
        return self._parse(
            response,
            flight_fabric=q.importer.flight_fabric,
            query=q
        )

    def validate_query(self, query):
        if self._brand not in BRANDS_TO_COUNTRY_COMBINATIONS:
            raise QueryIsNotValid('Brand "%s" is not allowed.' % self._brand)
        if get_request_country(query, self._brand) not in BRANDS_TO_COUNTRY_COMBINATIONS.get(self._brand):
            raise QueryIsNotValid('National version "%s" is not allowed for "%s" brand.' % (query.national_version, self._brand))

    def _get_data(self, tracker, query):
        url = self._query_builder.build(query)
        self._logger.info('Send request: %r', url)

        r = tracker.wrap_request(
            requests.get,
            url,
            auth=(self._login, self._password),
            verify=False
        )

        r.raise_for_status()
        if r.status_code == 204:  # Not found
            return {}

        return r.json()

    def _parse(self, response, flight_fabric, query):
        offers = response.get('offers', {})
        if not offers:
            return []

        currency_code = offers.get('currency', {}).get('code')
        if not currency_code:
            self._logger.warning('No currency code')
            return []

        variants = []
        for variant_description in sleep_every(offers.get('offer', [])):
            price = variant_description['price']
            v = Variant()

            bound = variant_description['bound']
            baggage = self._get_baggage(variant_description)
            v.forward.segments = self._make_segment_from_bound(flight_fabric, bound[0], baggage)
            v.backward.segments = self._make_segment_from_bound(flight_fabric, bound[1], baggage) if query.date_backward else []

            v.tariff = Price(float(price), currency_code)

            v.klass = query.klass

            url = variant_description['url']
            v.url = url
            v.order_data = {
                'url': url,
                'm_url': url + '&ibe.m=true',
            }

            variants.append(v)

        return variants

    def _make_segment_from_bound(self, flight_fabric, segments_description, baggage):
        return [
            flight_fabric.create(
                station_from_iata=item['from']['iata'],
                station_to_iata=item['to']['iata'],
                local_departure=self._make_datetime(item['from']['date'], item['from']['time']),
                local_arrival=self._make_datetime(item['to']['date'], item['to']['time']),
                company_iata=item['marketingCarrier'],
                pure_number=item['flight'],
                baggage=baggage,
            )
            for item in segments_description['segment']
        ]

    @staticmethod
    def _make_datetime(date, time):
        """
        Construct datetime
        :param date %Y-%m-%d
        :param time %H:%M
        :return datetime
        """

        return datetime(
            int(date[0:4]), int(date[5:7]), int(date[8:10]),
            int(time[0:2]), int(time[3:5]),
        )

    def _get_baggage(self, variant_description):
        baggage = variant_description.get('freeBaggage')
        if not baggage:
            return Baggage.from_partner()

        baggage_type = baggage.get('type')
        amount = baggage.get('amount')
        if baggage_type == 'P':
            return Baggage.from_partner(pieces=amount)

        if baggage_type == 'W':
            unit = baggage.get('unit')

            if unit == 'K':
                return Baggage.from_partner(weight=amount)

            if unit == 'P':
                return Baggage.from_partner(weight=int(amount * ONE_POUND_IN_KILO))

            self._logger.error('Unknown baggage unit %s in %r', unit, baggage)
            return Baggage.from_partner()

        self._logger.error('Unknown baggage type %s in %r', baggage_type, baggage)


def process_order_data(order_data):
    ext_params = _get_ext_params(order_data['qid'])

    if 'm_url' in order_data:
        return {
            'url': url_complement_missing(order_data['url'], ext_params),
            'm_url': url_complement_missing(order_data['m_url'], ext_params),
        }
    return url_complement_missing(order_data['url'], ext_params)


def _get_ext_params(qid):
    ext_params = {}

    service = get_service_by_qid(qid)

    if service.is_mobile:
        ext_params['ext-src'] = 'touch'
    else:
        ext_params['ext-src'] = 'desktop'

    return ext_params
