# -*- coding: utf-8 -*-
import itertools
from datetime import datetime, timedelta
from urlparse import urljoin
from uuid import uuid4

import requests
from retrying import retry

from travel.avia.ticket_daemon.ticket_daemon.api.flights import Variant, FlightFabric, Segment
from travel.avia.ticket_daemon.ticket_daemon.api.query import QueryIsNotValid
from travel.avia.ticket_daemon.ticket_daemon.daemon.utils import sleep_every
from travel.avia.ticket_daemon.ticket_daemon.lib.currency import Price
from travel.avia.ticket_daemon.ticket_daemon.lib.http import url_complement_missing
from travel.avia.ticket_daemon.ticket_daemon.lib.tracker import QueryTracker
from travel.avia.ticket_daemon.ticket_daemon.lib.partner_secret_storage import partner_secret_storage
from travel.avia.ticket_daemon.ticket_daemon.lib.baggage import Baggage
from travel.avia.library.python.ticket_daemon.memo import CacheInMemcache, memoize
from travel.avia.ticket_daemon.ticket_daemon.api.cache import shared_cache


FLYONE_LOGIN = 'YandexApi'
API_URL = 'https://otaapi.flyone.eu'
SEARCH_URL = urljoin(API_URL, '/api/OTA/search-flights')
LOGIN_URL = urljoin(API_URL, '/api/OTA/login')
FLYONE_PASSWORD = partner_secret_storage.get(importer_name='flyone', namespace='PASSWORD')
DATETIME_FORMAT = '%Y-%m-%dT%H:%M:%S'
DEEPLINK_DATE_FORMAT = '%d-%b-%Y'
DEEPLINK_BASE_URL = 'https://bookings.flyone.eu/FlightResult'
FLYONE_MARKER_PREFIX = '1002-'


class FareFamily(object):
    STANDARD = 'STANDARD'
    LOYAL = 'LOYAL'
    ADVANTAGE = 'ADVANTAGE'


FARE_FAMILIES_MAP = {
    'ST': FareFamily.STANDARD,
    'LO': FareFamily.LOYAL,
    'AD': FareFamily.ADVANTAGE,
}
REVERSED_FARE_FAMILIES_MAP = {v: k for k, v in FARE_FAMILIES_MAP.items()}

BAGGAGE_BY_FARE_FAMILY = {
    FareFamily.STANDARD: Baggage.from_partner(0),
    FareFamily.LOYAL: Baggage.from_partner(weight=20),
    FareFamily.ADVANTAGE: Baggage.from_partner(weight=20),
}


def validate_query(q):
    if q.klass != 'economy':
        raise QueryIsNotValid('Only economy requests allowed')


@QueryTracker.init_query
def query(tracker, q):
    token = get_token(tracker)
    json_content = get_data(tracker, q, token)
    variants = list(generate_variants(json_content, q))
    return variants


class FlyoneNoTokenError(Exception):
    pass


@memoize(lambda tracker: 'flyone', CacheInMemcache(shared_cache, 600, suffix='token'))
@retry(
    retry_on_exception=lambda e: isinstance(e, FlyoneNoTokenError),
    stop_max_attempt_number=3,
    wait_exponential_multiplier=100,
)
def get_token(tracker):
    response = tracker.wrap_request(
        requests.post,
        LOGIN_URL,
        json={'userName': FLYONE_LOGIN, 'password': FLYONE_PASSWORD},
        headers={'Content-Type': 'application/json'}
    ).json()
    if not response.get('token'):
        raise FlyoneNoTokenError('Failed to get Flyone token')
    return response['token']


def build_search_params(q):
    search_params = {
        'LanguageCode': 'en-GB',
        'IpAddress': '',
        'currencyCode': 'EUR',
        'reservationType': 1,
        'searchCriteria': {
            'PaxInfo': list(itertools.chain(
                ({'PaxType': 1, 'PaxKey': 'Adult{}'.format(i + 1)} for i in range(q.passengers.get('adults', 0))),
                ({'PaxType': 2, 'PaxKey': 'Child{}'.format(i + 1)} for i in range(q.passengers.get('children', 0))),
                ({'PaxType': 4, 'PaxKey': 'Infant{}'.format(i + 1)} for i in range(q.passengers.get('infants', 0))),
            )),
            'JourneyInfo': {
                'JourneyType': 2 if q.date_backward else 1,
                'RouteInfo': [
                    {
                        'DepCity': q.iata_from.encode('utf-8'),
                        'ArrCity': q.iata_to.encode('utf-8'),
                        'TravelDate': q.date_forward.strftime('%Y-%m-%d'),
                    }
                ]
            }
        }
    }
    if q.date_backward:
        search_params['searchCriteria']['JourneyInfo']['RouteInfo'].append({
            'DepCity': q.iata_to.encode('utf-8'),
            'ArrCity': q.iata_from.encode('utf-8'),
            'TravelDate': q.date_backward.strftime('%Y-%m-%d'),
        })
    return search_params


def get_data(tracker, q, token):
    r = tracker.wrap_request(
        requests.post,
        SEARCH_URL,
        json=build_search_params(q),
        headers={
            'Authorization': 'Bearer ' + token,
            'Content-Type': 'application/json',
        }
    )

    return r.json()


def generate_variants(result, q):
    available_flights = result.get('availableFlights')
    if not available_flights:
        return

    forward_flights = [f for f in available_flights if f['itinerary']['flightDirection'] == 1]
    backward_flights = [{}]
    if q.date_backward:
        backward_flights = [f for f in available_flights if f['itinerary']['flightDirection'] == 2]

    for forward_flight, backward_flight in sleep_every(itertools.product(forward_flights, backward_flights)):
        forward_variants_with_fare = _generate_variants_with_fares(forward_flight)
        backward_variants_with_fare = _generate_variants_with_fares(backward_flight)
        combination = itertools.product(forward_variants_with_fare, backward_variants_with_fare)
        for (forward_variant, forward_fare), (backward_variant, backward_fare) in sleep_every(combination):
            v = Variant()
            v.klass = q.klass
            price = float(forward_fare['priceInfo']['total'])
            v.forward.segments = [
                parse_segment(segment, forward_fare, q.importer.flight_fabric, q.klass)
                for segment in forward_variant['segments']
            ]
            if backward_fare:
                price += backward_fare['priceInfo']['total']
                v.backward.segments = [
                    parse_segment(segment, backward_fare, q.importer.flight_fabric, q.klass)
                    for segment in backward_variant['segments']
                ]
                if bad_cross_segments(v):
                    continue

            v.tariff = Price(price, forward_fare.get('currencyCode'))
            v.url = _build_deeplink(v, q)
            v.order_data = {'url': v.url}

            yield v


def _generate_variants_with_fares(raw_flight):
    if not raw_flight or not raw_flight.get('fareGroupInfo'):
        return [({}, {})]
    itinerary = raw_flight['itinerary']
    fares = [fare for fare in raw_flight['fareGroupInfo'] if itinerary['flightKey'] in fare['flightKeyRefs']]
    return itertools.product([itinerary], fares)


def parse_segment(segment, fare, flight_fabric, klass):
    # type: (dict, dict, FlightFabric, str) -> Segment
    cabin = [c for c in fare['cabin'] if c['segmentRefKey'] == segment['segmentKey']][0]
    fare_family = FARE_FAMILIES_MAP.get(cabin['cabinClassCode'], FareFamily.STANDARD)
    return flight_fabric.create(
        station_from_iata=segment['depCode'],
        station_to_iata=segment['arrCode'],
        local_departure=datetime.strptime(segment['depDateTime'], DATETIME_FORMAT),
        local_arrival=datetime.strptime(segment['arrDateTime'], DATETIME_FORMAT),
        company_iata=segment['mktAirlineCode'],
        pure_number=segment['flightNumber'],
        klass=klass,
        fare_code=cabin['fareBasisCode'],
        fare_family=fare_family,
        baggage=BAGGAGE_BY_FARE_FAMILY.get(fare_family),
    )


def _build_deeplink(v, q):
    deeplink_params = {
        'depCity': v.forward.segments[0].station_from_iata,
        'arrCity': v.forward.segments[-1].station_to_iata,
        'startDate': v.forward.segments[0].local_departure.strftime(DEEPLINK_DATE_FORMAT),
        'adult': q.passengers.get('adults', 1),
        'child': q.passengers.get('children', 0),
        'infant': q.passengers.get('infants', 0),
        'ob': '_'.join([
            f.number.replace(' ', '') for f in v.forward.segments
        ]) + '-' + REVERSED_FARE_FAMILIES_MAP[v.forward.segments[0].fare_family]
    }

    if q.date_backward:
        deeplink_params['endDate'] = v.backward.segments[-1].local_departure.strftime(DEEPLINK_DATE_FORMAT)
        deeplink_params['ib'] = '_'.join([
            f.number.replace(' ', '') for f in v.backward.segments
        ]) + '-' + REVERSED_FARE_FAMILIES_MAP[v.backward.segments[0].fare_family]

    return url_complement_missing(DEEPLINK_BASE_URL, deeplink_params)


def bad_cross_segments(v):
    return (
        v.backward.segments and
        v.forward.segments[-1].local_arrival and
        v.backward.segments[0].local_departure and (
            v.forward.segments[-1].local_arrival + timedelta(hours=1) >
            v.backward.segments[0].local_departure
        )
    )


def book(order_data):
    return {'url': order_data['url']}


def generate_marker():
    # type: () -> str
    return FLYONE_MARKER_PREFIX + _generate_alphanumeric_marker()


def _generate_alphanumeric_marker():
    # type: () -> str
    return str(uuid4()).replace('-', '')
