# -*- coding: utf-8 -*-
import json
import re
from datetime import datetime
from logging import getLogger
from urllib import urlencode

import requests

from travel.avia.ticket_daemon.ticket_daemon.api.flights import SERVICE_CLASS_PRIORITIES, Variant
from travel.avia.ticket_daemon.ticket_daemon.daemon.utils import sleep_every
from travel.avia.ticket_daemon.ticket_daemon.lib.baggage import Baggage
from travel.avia.ticket_daemon.ticket_daemon.lib.currency import Price
from travel.avia.ticket_daemon.ticket_daemon.lib.decorators import elementwise
from travel.avia.ticket_daemon.ticket_daemon.lib.tracker import QueryTracker
from travel.avia.ticket_daemon.ticket_daemon.lib.partner_secret_storage import partner_secret_storage

log = getLogger(__name__)

KLASSSES_MAPPING = {
    'ECONOM': 'economy',
    'BUSINESS': 'business',
    'FIRST': 'first',
}

SERVICE_URL = 'http://service.just.travel/'
LOGIN = 'yandex.avia@just.travel'
PASSWORD = partner_secret_storage.get(
    importer_name='justtravel', namespace='PASSWORD'
)

BAGGAGE_FORMAT = re.compile(r'(\d+)(\w+)')


def validate_query(q):
    q.validate_passengers(adults=5, children=5, count=9)


@QueryTracker.init_query
def query(tracker, q):
    variants_list, principal_id = get_data(tracker, q)
    variants = parse_response(variants_list, principal_id, q)

    return variants


def get_data(tracker, q):
    session = requests.Session()

    r = tracker.wrap_request(
        session.post,
        SERVICE_URL + 'json/authentication/login',
        headers={
            'Content-Type': 'application/json; charset=utf-8',
        },
        data=json.dumps({
            'username': LOGIN,
            'password': PASSWORD,
        }).encode('utf-8')
    )

    principal_id = r.json()['principalId']

    params = {
        'tripRoute': [{
            'departureAirportIataCode': q.iata_from,
            'arrivalAirportIataCode': q.iata_to,
            'departureDate': q.date_forward.strftime('%Y-%m-%d'),
        }],
        'adultCount': q.passengers.get('adults', 0),
        'childrenCount': q.passengers.get('children', 0),
        'infantCount': q.passengers.get('infants', 0),
        'serviceClass': {'economy': 'ECONOM', 'business': 'BUSINESS'}[q.klass],
        'maxStopsNumber': 2,
    }

    if q.date_backward:
        params['tripRoute'].append({
            'departureAirportIataCode': q.iata_to,
            'arrivalAirportIataCode': q.iata_from,
            'departureDate': q.date_backward.strftime('%Y-%m-%d'),
        })

    r = tracker.wrap_request(
        session.post,
        SERVICE_URL + 'json/avia/getTripList',
        headers={
            'User-Token': principal_id,
            'Content-Type': 'application/json; charset=utf-8',
        },
        data=json.dumps(params).encode('utf-8')
    )

    return r.json(), principal_id


def min_class(service_classes):
    return min(service_classes, key=lambda c: SERVICE_CLASS_PRIORITIES[c])


def parse_response(variants_list, principal_id, q):
    variants = []

    for v_tag in sleep_every(variants_list):
        v = Variant()

        routes = v_tag['aviaSegments']
        v.forward.segments = parse_flights(
            routes[0]['aviaSegmentLegs'], q.importer.flight_fabric)

        if q.date_backward and len(routes) > 1:
            v.backward.segments = parse_flights(
                routes[1]['aviaSegmentLegs'], q.importer.flight_fabric)

        variant_segments = v.forward.segments + (v.backward.segments or [])

        v.klass = min_class([s.klass for s in variant_segments])

        currency = v_tag['priceInfo']['currency']

        if currency == 'RUB':
            currency = 'RUR'

        price_val = sum([
            float(p['serviceCost']) * int(p['countPassengerByType'])
            for p in v_tag['priceInfo']['priceByPassengers']
        ])

        v.tariff = Price(price_val, currency)

        v.variant = v_tag['flightId']
        v.order_data = {
            'variant': v.variant,
            'principal': principal_id,
        }

        variants.append(v)

    return variants


def parse_flights(flights_tags, flight_fabric):
    return filter(None, parse_flight(flights_tags, flight_fabric))


@elementwise
def parse_flight(f_tag, flight_fabric):
    try:
        return _parse_flight(f_tag, flight_fabric)

    except Exception:
        log.warning(u'f_tag: %r', f_tag)
        raise


def get_baggage(f_tag):
    try:
        baggage = f_tag['baggage']
        if baggage == 'N/A':
            return Baggage.from_partner()
        update_info = {}
        for count, desc in BAGGAGE_FORMAT.findall(baggage):
            if desc == 'pc':
                update_info['pieces'] = int(count)
            elif desc == 'kg':
                update_info['weight'] = int(count)
            else:
                log.info('Unknown baggage format %s', baggage)
                return Baggage.from_partner()
        return Baggage.from_partner(**update_info)
    except Exception, exc:
        log.info('Baggage parsing exception: {} on element {}'.format(exc, f_tag))
    return Baggage.from_partner()


def _parse_flight(f_tag, flight_fabric):
    return flight_fabric.create(
        station_from_iata=f_tag['departureAirportIataCode'],
        station_to_iata=f_tag['arrivalAirportIataCode'],
        local_departure=datetime.strptime(f_tag['departureTime'],
                                          '%Y-%m-%dT%H:%M:%S'),
        local_arrival=datetime.strptime(f_tag['arrivalTime'],
                                        '%Y-%m-%dT%H:%M:%S'),
        klass=KLASSSES_MAPPING[f_tag['serviceClass']],
        company_iata=f_tag['companyIataCode'],
        pure_number=f_tag['flightNumber'],
        baggage=get_baggage(f_tag),
    )


def book(order_data):
    return 'http://www.just.travel/avia?' + urlencode({
        'flightId': order_data['variant'],
        'principal_id': order_data['principal'],
    })
