# -*- coding: utf-8 -*-
from datetime import datetime

import requests

from travel.avia.ticket_daemon.ticket_daemon.lib.baggage import Baggage
from travel.avia.ticket_daemon.ticket_daemon.lib.decorators import pipe
from travel.avia.ticket_daemon.ticket_daemon.api.flights import Variant
from travel.avia.ticket_daemon.ticket_daemon.daemon.utils import sleep_every
from travel.avia.ticket_daemon.ticket_daemon.lib.currency import Price
from travel.avia.ticket_daemon.ticket_daemon.lib.tracker import QueryTracker

API_URL = 'https://api.skypicker.com/flights'
API_PARTNER = 'yandexflights'
CURRENCIES_MAP = {'ru': 'RUB', 'ua': 'UAH', 'tr': 'TRY', 'com': 'EUR', 'kz': 'KZT'}
CURRENCIES = frozenset((
    'AED', 'AFN', 'ALL', 'AMD', 'ANG', 'AOA', 'ARS', 'AUD', 'AWG', 'AZN', 'BAM', 'BBD', 'BDT', 'BGN', 'BHD', 'BIF',
    'BMD', 'BND', 'BOB', 'BRL', 'BSD', 'BTC', 'BTN', 'BWP', 'BYR', 'BZD', 'CAD', 'CDF', 'CHF', 'CLF', 'CLP', 'CNY',
    'COP', 'CRC', 'CUC', 'CUP', 'CVE', 'CZK', 'DJF', 'DKK', 'DOP', 'DZD', 'EEK', 'EGP', 'ERN', 'ETB', 'EUR', 'FJD',
    'FKP', 'GBP', 'GEL', 'GGP', 'GHS', 'GIP', 'GMD', 'GNF', 'GTQ', 'GYD', 'HKD', 'HNL', 'HRK', 'HTG', 'HUF', 'IDR',
    'ILS', 'IMP', 'INR', 'IQD', 'IRR', 'ISK', 'JEP', 'JMD', 'JOD', 'JPY', 'KES', 'KGS', 'KHR', 'KMF', 'KPW', 'KRW',
    'KWD', 'KYD', 'KZT', 'LAK', 'LBP', 'LKR', 'LRD', 'LSL', 'LTL', 'LVL', 'LYD', 'MAD', 'MDL', 'MGA', 'MKD', 'MMK',
    'MNT', 'MOP', 'MRO', 'MTL', 'MUR', 'MVR', 'MWK', 'MXN', 'MYR', 'MZN', 'NAD', 'NGN', 'NIO', 'NOK', 'NPR', 'NZD',
    'OMR', 'PAB', 'PEN', 'PGK', 'PHP', 'PKR', 'PLN', 'PYG', 'QAR', 'QUN', 'RON', 'RSD', 'RUB', 'RWF', 'SAR', 'SBD',
    'SCR', 'SDG', 'SEK', 'SGD', 'SHP', 'SLL', 'SOS', 'SRD', 'STD', 'SVC', 'SYP', 'SZL', 'THB', 'TJS', 'TMT', 'TND',
    'TOP', 'TRY', 'TTD', 'TWD', 'TZS', 'UAH', 'UGX', 'USD', 'UYU', 'UZS', 'VEF', 'VND', 'VUV', 'WST', 'XAF', 'XAG',
    'XAU', 'XCD', 'XDR', 'XOF', 'XPD', 'XPF', 'XPT', 'YER', 'ZAR', 'ZMK', 'ZMW', 'ZWL',
))
LOCALES = frozenset((
    'ae', 'ag', 'ar', 'at', 'au', 'be', 'bg', 'bh', 'br', 'by', 'ca', 'ca-fr', 'ch', 'cl', 'cn', 'co', 'ct', 'cz',
    'da', 'de', 'dk', 'ec', 'ee', 'el', 'en', 'es', 'fi', 'fr', 'gb', 'gr', 'hk', 'hr', 'hu', 'id', 'ie', 'il', 'in',
    'is', 'it', 'ja', 'jo', 'jp', 'ko', 'kr', 'kw', 'kz', 'lt', 'mx', 'my', 'nl', 'no', 'nz', 'om', 'pe', 'ph', 'pl',
    'pt', 'qa', 'ro', 'rs', 'ru', 'sa', 'se', 'sg', 'sk', 'sr', 'sv', 'th', 'tr', 'tw', 'ua', 'uk', 'us', 'vn', 'za',
))
allowed_klasses = frozenset(('economy',))  # т.к. в АПИ нельзя передавать класс, то будет только economy
PARTNER_MARKET_MAP = {'com': 'ie'}


def validate_query(q):
    """
    :type q: ticket_daemon.api.query.Query
    """
    q.validate_klass(allowed_klasses)


def search_params(q):
    """
    EXAMPLE: https://api.skypicker.com/flights?flyFrom=CZ&to=porto&dateFrom=08%2F08%2F2017&dateTo=08%2F09%2F2017&longitudeFrom=14.0000&latitudeFrom=50.2000&radiusFrom=200&longitudeTo=14.0000&latitudeTo=50.2000&radiusTo=200&daysInDestinationFrom=2&daysInDestinationTo=14&returnFrom=08%2F08%2F2017&returnTo=08%2F09%2F2017&maxFlyDuration=6&typeFlight=oneway&oneforcity=0&one_per_date=0&passengers=1&adults=1&children=0&infants=0&flyDays=%5B0%2C1%2C2%2C3%2C4%2C5%2C6%5D&flyDaysType=departure&returnFlyDays=%5B0%2C1%2C2%2C3%2C4%2C5%2C6%5D&returnFlyDaysType=departure&onlyWorkingDays=0&onlyWeekends=0&directFlights=0&partner=picky&partner_market=us&v=2&xml=0&curr=EUR&locale=en&price_from=1&price_to=10000&dtimefrom=00%3A00&dtimeto=00%3A00&atimefrom=00%3A00&atimeto=00%3A00&returndtimefrom=00%3A00&returndtimeto=00%3A00&returnatimefrom=00%3A00&returnatimeto=00%3A00&stopoverfrom=00%3A00&stopoverto=00%3A00&maxstopovers=2&connectionsOnDifferentAirport=1&returnFromDifferentAirport=1&returnToDifferentAirport=1&ref_tasks=null&innerLimit=300&selectedAirlines=null&selectedAirlinesExclude=False&selectedStopoverAirports=null&selectedStopoverAirportsExclude=False&booking_token=hashed%20data&offset=0&limit=30&sort=price&asc=1
    :type q: ticket_daemon.api.query.Query
    """  # noqa
    dt_format = '%d/%m/%Y'
    date_from = q.date_forward.strftime(dt_format)
    params = {
        'partner': API_PARTNER,
        'flyFrom': q.iata_from,
        'to': q.iata_to,
        'dateFrom': date_from,
        'dateTo': date_from,  # Киви умеет принимать даты вылета в диапазоне, но нам пока этого не нужно
        'adults': q.passengers.get('adults', 0),
        'children': q.passengers.get('children', 0),
        'infants': q.passengers.get('infants', 0),
        'curr': CURRENCIES_MAP.get(q.national_version, CURRENCIES_MAP['ru']),
        'locale': q.lang if q.lang in LOCALES else 'ru',
        'partner_market': PARTNER_MARKET_MAP.get(q.national_version, q.national_version),
    }
    if q.date_backward:
        date_to = q.date_backward.strftime(dt_format)
        params['returnFrom'] = date_to
        params['returnTo'] = date_to
    return params


@QueryTracker.init_query
def query(tracker, q):
    data = get_data(tracker, q)

    variants = parse_response(data, q)

    return variants


def get_data(tracker, q):
    return tracker.wrap_request(
        requests.get,
        API_URL,
        params=search_params(q),
    ).json()


def get_baggage(variant):
    if variant["bags_price"].get("1") == 0:
        return Baggage.from_partner(weight=variant['baglimit']["hold_weight"])
    else:
        return Baggage.from_partner(included=False)


@pipe(list)
def parse_response(data, q):
    for variant in sleep_every(data['data']):
        v = Variant()
        v.klass = q.klass
        routes = filter(lambda r: r['return'] == 0, variant['route'])
        baggage = get_baggage(variant)
        v.forward.segments = parse_segments(routes, q.importer.flight_fabric, baggage)
        if q.date_backward:
            routes = filter(lambda r: r['return'] == 1, variant['route'])
            v.backward.segments = parse_segments(routes, q.importer.flight_fabric, baggage)
        v.tariff = Price(float(variant['price']), currency=data['currency'])
        v.order_data = {'url': variant['deep_link']}
        yield v


@pipe(list)
def parse_segments(segments, flight_fabric, baggage):
    for segment in segments:
        yield flight_fabric.create(
            company_iata=segment['airline'],
            pure_number=segment['flight_no'],
            station_from_iata=segment['flyFrom'],
            station_to_iata=segment['flyTo'],
            local_departure=datetime.utcfromtimestamp(segment['dTime']),
            local_arrival=datetime.utcfromtimestamp(segment['aTime']),
            baggage=baggage,
            selfconnect=segment.get('guarantee'),
            fare_code=segment.get('fare_basis'),
        )
