# -*- coding: utf-8 -*-
from datetime import datetime
from logging import getLogger

import requests
from lxml import etree

from travel.avia.ticket_daemon.ticket_daemon.api.flights import Variant, Segment, FlightFabric
from travel.avia.ticket_daemon.ticket_daemon.api.query import Query
from travel.avia.ticket_daemon.ticket_daemon.daemon.utils import sleep_every
from travel.avia.ticket_daemon.ticket_daemon.lib.baggage import BaggageParser
from travel.avia.ticket_daemon.ticket_daemon.lib.currency import Price
from travel.avia.ticket_daemon.ticket_daemon.lib.tracker import QueryTracker
from travel.avia.ticket_daemon.ticket_daemon.lib.partner_secret_storage import partner_secret_storage


log = getLogger(__name__)
baggage_parser = BaggageParser(logger=log)

API_URL = 'https://api.kupibilet.ru/d2c66cbdcacd2dc70db57c275b13a013'
PARTNER = 'yandex'
PASSWORD = partner_secret_storage.get(
    importer_name='kupibilet', namespace='PASSWORD'
)

PARTNER_KZ = 'yandex_kz'
PASSWORD_KZ = partner_secret_storage.get(
    importer_name='kupibilet_kz', namespace='PASSWORD'
)


@QueryTracker.init_query
def query(tracker, q):
    # type: (QueryTracker, Query) -> list
    search_params = get_search_params(q)
    r = tracker.wrap_request(
        requests.get,
        API_URL,
        params=search_params
    )
    variants = parse_response(r.content, q)

    return variants


def get_search_params(q):
    # type: (Query) -> dict
    if q.national_version == 'kz':
        partner, password = PARTNER_KZ, PASSWORD_KZ
    else:
        partner, password = PARTNER, PASSWORD

    params = {
        'from': q.iata_from,
        'to': q.iata_to,
        'date1': q.date_forward.strftime('%Y-%m-%d'),
        'adults': q.passengers.get('adults', 0),
        'children': q.passengers.get('children', 0),
        'infants': q.passengers.get('infants', 0),
        'cabin': {'economy': 'Y', 'business': 'C'}[q.klass],
        'locale': q.lang,
        'partner': partner,
        'password': password,
    }

    if q.date_backward:
        params['date2'] = q.date_backward.strftime('%Y-%m-%d')

    return params


def extract_datetime(tag, date_attr, time_attr):
    raw_date = tag.findtext(date_attr)
    raw_time = tag.findtext(time_attr)

    return datetime(
        int(raw_date[0:4]), int(raw_date[5:7]), int(raw_date[8:10]),
        int(raw_time[0:2]), int(raw_time[3:5]),
    )


def parse_response(xml, q):
    # type: (str, Query) -> list
    variants = []
    tree = etree.fromstring(xml)
    fabric = q.importer.flight_fabric

    for variant in sleep_every(tree.xpath('/variants/variant')):
        try:
            v = Variant()
            v.klass = q.klass

            segments = variant.xpath('segment')

            v.forward.segments = [parse_flight(flight, fabric)
                                  for flight in segments[0]]

            if bool(q.date_backward and len(segments) > 1):
                v.backward.segments = [parse_flight(flight, fabric)
                                       for flight in segments[1]]

            v.tariff = Price(
                float(variant.findtext('price')),
                variant.findtext('currency').upper().replace('RUB', 'RUR')
            )
            v.order_data = {
                'url': variant.findtext('url'),
            }
            v.charter = variant.findtext('isCharter') == 'true'

            variants.append(v)
        except Exception as e:
            log.error('Error while parsing variant: %r', e)
    return variants


def parse_flight(flight, flight_fabric):
    # type: (etree._Element, FlightFabric) -> Segment

    return flight_fabric.create(
        station_from_iata=flight.findtext('departure'),
        station_to_iata=flight.findtext('arrival'),
        local_departure=extract_datetime(flight, 'departureDate', 'departureTime'),
        local_arrival=extract_datetime(flight, 'arrivalDate', 'arrivalTime'),
        company_iata=flight.findtext('operatingCarrier'),
        pure_number=flight.findtext('number'),
        baggage=baggage_parser.parse_from_string(flight.findtext('baggage')),
        fare_code=flight.findtext('fareCode'),
    )
