# -*- coding: utf-8 -*-
import logging
import urllib
from datetime import datetime

import requests
from lxml import etree

from travel.avia.ticket_daemon.ticket_daemon.daemon.utils import sleep_every
from travel.avia.ticket_daemon.ticket_daemon.lib.currency import Price
from travel.avia.ticket_daemon.ticket_daemon.api.flights import Variant
from travel.avia.ticket_daemon.ticket_daemon.lib.baggage import Baggage
from travel.avia.ticket_daemon.ticket_daemon.lib.tracker import QueryTracker
from travel.avia.ticket_daemon.ticket_daemon.lib.utils import skip_None_values
from travel.avia.ticket_daemon.ticket_daemon.lib.http import url_complement_missing


log = logging.getLogger(__name__)

MOIREIS_URL = 'https://www.moireis.ru/gate/yandex.php'
URL_PARAMS = {
    'utm_source': 'yandex.avia',
}

KLASS_MAP = {
    'economy': 'E',
    'business': 'B',
    'first': 'F',
}
REVERSE_KLASS_MAP = {v: k for k, v in KLASS_MAP.items()}


@QueryTracker.init_query
def query(tracker, q):
    xml = get_data(tracker, q)

    variants = list(parse_response(xml, q))

    return variants


def get_data(tracker, q):
    params = {
        'departure': q.iata_from,
        'arrival': q.iata_to,
        'date_forward': q.date_forward.strftime('%Y-%m-%d'),
        'date_backward': q.date_backward.strftime('%Y-%m-%d') if q.date_backward else None,
        'class': KLASS_MAP[q.klass],
        'adult': q.passengers.get('adults', 0),
        'child': q.passengers.get('children', 0),
        'infant': q.passengers.get('infants', 0)
    }

    r = tracker.wrap_request(
        requests.post,
        MOIREIS_URL,
        data=skip_None_values(params),
        verify=False,
    )

    return r.text.encode('utf-8')


def parse_response(xml, q):
    tree = etree.fromstring(xml)
    for vtag in sleep_every(tree.xpath('variant')):
        try:
            yield parse_variant(q, vtag)
        except Exception, exc:
            log.warn(u'Ошибка разбора варианта: %s', exc, exc_info=True)


def parse_variant(q, variant_tag):
    fare = variant_tag.find('fare')

    v = Variant()
    v.tariff = Price(
        float(fare.get('value')),
        fare.get('currency').upper().replace('RUB', 'RUR'),
    )
    v.charter = fare.get('charter') == 'true'
    v.klass = REVERSE_KLASS_MAP[fare.get('class')]

    v.forward.segments = list(parse_flights(
        variant_tag.xpath('./route_forward'),
        fare,
        q.importer.flight_fabric,
    ))

    if q.date_backward:
        v.backward.segments = list(parse_flights(
            variant_tag.xpath('./route_backward'),
            fare,
            q.importer.flight_fabric,
        ))

    v.url = url_complement_missing(
        urllib.unquote(variant_tag.get('url')),
        params=URL_PARAMS,
    )

    v.order_data = {
        'url': v.url,
    }

    return v


def parse_flights(route_tags, fare, flight_fabric):
    for route_tag in route_tags:
        yield flight_fabric.create(
            local_departure=parse_datetime(route_tag.get('departure_datetime')),
            local_arrival=parse_datetime(route_tag.get('arrival_datetime')),
            company_iata=route_tag.get('company_code'),
            station_from_iata=route_tag.get('departure_airport_code'),
            station_to_iata=route_tag.get('arrival_airport_code'),
            pure_number=route_tag.get('route_code'),
            baggage=get_baggage(route_tag, fare),
        )


def get_baggage(route_tag, fare):
    try:
        if fare.get('luggage') == 'true':
            return Baggage.from_partner(
                included=True,
                pieces=fare.get('pieces_of_luggage'),
                weight=fare.get('luggage_weight'),
            )
        else:
            return Baggage.from_partner(included=False)
    except Exception, exc:
        log.warn(
            'Baggage parsing exception: %s on element %s, fare %s',
            exc,
            route_tag,
            fare,
            exc_info=True,
        )

    return Baggage.from_partner()


def parse_datetime(s):
    return datetime(
        int(s[0:4]), int(s[5:7]), int(s[8:10]),
        int(s[11:13]), int(s[14:17]),
    )
