# -*- coding: utf-8 -*-
import base64
import cPickle
import re
import zlib
from datetime import datetime
from logging import getLogger
from urlparse import urlparse

import requests
from django.template import loader
from lxml import etree

from travel.avia.ticket_daemon.ticket_daemon.api.flights import Variant
from travel.avia.ticket_daemon.ticket_daemon.daemon.utils import sleep_every
from travel.avia.ticket_daemon.ticket_daemon.lib.currency import Price
from travel.avia.ticket_daemon.ticket_daemon.lib.http import url_complement_missing
from travel.avia.ticket_daemon.ticket_daemon.lib.tracker import QueryTracker
from travel.avia.ticket_daemon.ticket_daemon.lib.partner_secret_storage import partner_secret_storage

log = getLogger(__name__)

URL_TRACKER = {'utm_source': 'yandex.avia'}

SEARCH_URL = 'http://webservices.nabortu.ru/WebServices/NabortuXml.asmx?op=GetVariants'
PARTNERID = 'YANDEX.RU'
LOGIN = 'aygul2@yandex-team.ru'
PASSWORD = partner_secret_storage.get(importer_name='nabortu', namespace='PASSWORD')
REDIRECT_URL = 'https://new.nabortu.ru/Wait/?action=avia_calculate&mode=init'


@QueryTracker.init_query
def query(tracker, q):
    # Партнер умеет отдавать только классы эконом и бизнес
    if q.klass not in ['economy', 'business']:
        return []

    aviasearch_params = build_aviasearch_params(q)

    xml = get_data(tracker, q, aviasearch_params)

    variants = list(parse_response(xml, q, aviasearch_params))

    return variants


def build_aviasearch_params(q):
    return {
        'iata_from': q.iata_from,
        'iata_to': q.iata_to,
        'forward_date': q.date_forward.strftime('%Y-%m-%d'),
        'return_date': q.date_backward and q.date_backward.strftime('%Y-%m-%d') or '',
        'adults': q.passengers.get('adults', 0),
        'children': q.passengers.get('children', 0),
        'infants': q.passengers.get('infants', 0),
        'fare': {'economy': 'Economy', 'business': 'Business'}[q.klass],
    }


def build_xml(xml_template_file, params):
    query_xml = loader.render_to_string(xml_template_file, params)
    return ''.join([line.strip() for line in filter(None, query_xml.strip().split('\n'))])


def build_aviasearch_request_xml(params):
    return build_xml('partners/nabortu_aviasearch_request.xml', params)


def get_data(tracker, q, aviasearch_params):
    params = {
        'aviasearch_request_xml': build_aviasearch_request_xml(aviasearch_params),
        'partnerid': PARTNERID,
        'login': LOGIN,
        'password': PASSWORD,
    }

    query_xml = build_xml('partners/nabortu.xml', params)

    r = tracker.wrap_request(
        requests.post,
        SEARCH_URL,
        headers={
            'Content-Type': 'text/xml; charset=utf-8',
            'SOAPAction': 'http://www.nabortu.ru/GetVariants',
            'Host': urlparse(SEARCH_URL).netloc.split(':')[0],
        },
        data=query_xml.encode('utf-8')
    )

    return r.content


def pack_aviasearch_params(aviasearch_params):
    return base64.encodestring(cPickle.dumps(aviasearch_params))


def unpack_aviasearch_params(aviasearch_params):
    return cPickle.loads(base64.decodestring(aviasearch_params))


def pack_item(item):
    return base64.encodestring(zlib.compress(item))


def unpack_item(item):
    return zlib.decompress(base64.decodestring(item))


def parse_response(xml, q, aviasearch_params):
    tree = etree.fromstring(xml)
    aviasearch_params_packed = pack_aviasearch_params(aviasearch_params)
    re_xmlns = re.compile(r'xmlns\:?.*?\s', re.S | re.U | re.M | re.I)

    for e in sleep_every(tree.xpath('//SearchResult/Offers/Item')):

        v = Variant()
        v.forward.segments = list(parse_flights(
            e.xpath('Flights/Item[@SegmentId="0"]'), q.importer.flight_fabric))
        if q.date_backward:
            backward_items = e.xpath('Flights/Item[@SegmentId="1"]')
            if backward_items:
                v.backward.segments = list(parse_flights(
                    backward_items, q.importer.flight_fabric))

        v.tariff = Price(float(e.get('Price').replace(',', '.')))
        v.charter = e.get('charter') == 'true'

        v.klass = q.klass

        proposal = etree.tostring(e)
        proposal = re_xmlns.sub('', proposal)

        v.order_data = {
            'aviasearch_params': aviasearch_params_packed,
            'proposal': pack_item(proposal),
        }
        yield v


def parse_flights(items, flight_fabric):
    for item in items:
        yield flight_fabric.create(
            station_from_iata=item.get('Origin'),
            station_to_iata=item.get('Destination'),
            local_departure=datetime.strptime(
                item.get('Depart'), '%Y-%m-%dT%H:%M:%S'),
            local_arrival=datetime.strptime(
                item.get('Arrive'), '%Y-%m-%dT%H:%M:%S'),
            company_iata=item.get('Code'),
            fare_code=item.get('FareName'),
            pure_number=item.get('Num'),
        )


def book(order_data):
    aviasearch_params = unpack_aviasearch_params(
        order_data['aviasearch_params']
    )

    aviasearch_params_xml = \
        build_aviasearch_request_xml(aviasearch_params).encode('utf-8')

    proposal_xml = unpack_item(order_data['proposal'])

    postdata = {
        'login': LOGIN,
        'password': PASSWORD,
        'PartnerId': PARTNERID,
        'RequestParameters': base64.encodestring(aviasearch_params_xml),
        'Proposal': base64.encodestring(proposal_xml),
    }

    return url_complement_missing(REDIRECT_URL, URL_TRACKER), postdata
