# -*- coding: utf-8 -*-
from datetime import datetime
from logging import getLogger

import requests
from lxml import etree

from travel.avia.ticket_daemon.ticket_daemon.api.flights import Variant
from travel.avia.ticket_daemon.ticket_daemon.api.query import QueryIsNotValid
from travel.avia.ticket_daemon.ticket_daemon.daemon.utils import sleep_every
from travel.avia.ticket_daemon.ticket_daemon.lib.baggage import BaggageParser
from travel.avia.ticket_daemon.ticket_daemon.lib.currency import Price
from travel.avia.ticket_daemon.ticket_daemon.lib.decorators import pipe
from travel.avia.ticket_daemon.ticket_daemon.lib.http import update_query_string
from travel.avia.ticket_daemon.ticket_daemon.lib.tracker import QueryTracker
from travel.avia.ticket_daemon.ticket_daemon.lib.utils import skip_None_values
from travel.avia.ticket_daemon.ticket_daemon.lib.partner_secret_storage import partner_secret_storage

log = getLogger(__name__)
baggage_parser = BaggageParser(log)

SEARCH_URL = 'https://api.nebo.travel/partners/yandex/flights'
LOGIN = 'yandex'

KLASS_MAP = {'economy': 'Y', 'business': 'C', 'first': 'F'}
REVERSED_KLASS_MAP = {KLASS_MAP[k]: k for k in KLASS_MAP}


def validate_query(q):
    if not all([hasattr(q, 'iata_real_to'), hasattr(q, 'iata_real_from')]):
        raise QueryIsNotValid('Only IATA codes allowed: %s %s' % (
            q.iata_from.encode('utf-8'),
            q.iata_to.encode('utf-8')
        ))


@QueryTracker.init_query
def query(tracker, q):
    tree = get_data(tracker, q)
    variants = parse_response(tree, q)

    return variants


def build_search_params(q):
    return skip_None_values({
        'from': q.iata_from.encode('utf-8'),
        'to': q.iata_to.encode('utf-8'),
        'date1': q.date_forward.strftime('%Y-%m-%d'),
        'date2': q.date_backward and q.date_backward.strftime('%Y-%m-%d') or None,
        'adults': q.passengers.get('adults', 0),
        'children': q.passengers.get('children', 0),
        'infants': q.passengers.get('infants', 0),
        'cabin': KLASS_MAP[q.klass],
        'partner': LOGIN,
        'password': partner_secret_storage.get(
            importer_name='nebotravel', namespace='PASSWORD'
        ),
    })


def get_data(tracker, q):
    r = tracker.wrap_request(
        requests.get,
        update_query_string(
            SEARCH_URL,
            build_search_params(q)
        ),
        verify=False,
    )

    tree = etree.fromstring(r.content)
    return tree


def parse_response(tree, q):
    variants = []

    for variant in sleep_every(list(tree.xpath('variant'))):
        try:
            v = Variant()

            v.forward.segments = parse_segments(
                variant.xpath('segment')[0], q.importer.flight_fabric
            )

            if q.date_backward:
                v.backward.segments = parse_segments(
                    variant.xpath('segment')[1], q.importer.flight_fabric
                )

            v.tariff = Price(float(variant.findtext('price')), variant.findtext('currency').upper())
            v.klass = q.klass
            is_charter = variant.findtext('isCharter')
            if is_charter not in ['true', 'false']:
                raise ValueError('isCharter should be in ["true", "false"]. Found {} instead.'.format(is_charter))
            v.charter = is_charter == 'true'
            v.url = variant.findtext('url')
            v.order_data = {'url': v.url}

            variants.append(v)
        except (ValueError, KeyError, TypeError) as e:
            log.error('Failed to parse variant: %s Error message: %s',
                      etree.tostring(variant, pretty_print=True),
                      e.message)
    return variants


DT_FMT = '%Y-%m-%d%H:%M'


@pipe(list)
def parse_segments(items, flight_fabric):
    for item in items:
        yield flight_fabric.create(
            station_from_iata=item.findtext('departure'),
            station_to_iata=item.findtext('arrival'),
            local_departure=datetime.strptime(
                '{}{}'.format(item.findtext('departureDate'), item.findtext('departureTime')), DT_FMT
            ),
            local_arrival=datetime.strptime(
                '{}{}'.format(item.findtext('arrivalDate'), item.findtext('arrivalTime')), DT_FMT
            ),
            company_iata=item.findtext('operatingCarrier'),
            pure_number=item.findtext('number'),
            fare_code=item.findtext('fareCode'),
            klass=REVERSED_KLASS_MAP[item.findtext('cabin')],
            baggage=baggage_parser.parse_from_string(item.findtext('baggage'))
        )
